/**
 * linked list containing Nodes
 * @author Lukas Armborst, University of Twente
 * @year 2021
 */

final class NodeList {
    Node node;
    NodeList next;


    /********
     Resource
     ********/

    /*@
    /// recursive resource to access all Nodes in this list
    public static resource list_perm(NodeList nl)
        = nl != null
            ==> (Perm(nl.node, write)** Perm(nl.next, write) 
                 ** nl.node != null 
                 ** Node.node_perm(nl.node, write)
                 ** list_perm(nl.next))
                 ;
    @*/


    /****************
     Helper Functions
     ****************/

    /*@
    /// number of Nodes in this list
        requires [read]list_perm(nl);
        ensures nl != null ==> \result > 0;
        ensures nl == null ==> \result == 0;
    pure static int size (NodeList nl) 
        = nl == null ? 0
                     : \unfolding [read]list_perm(nl) \in (1 + size(nl.next));
    @*/
    

    /*@
    /// turn linked list of Nodes into a sequence of Nodes
        requires [read]list_perm(nl);
        ensures |\result| == size(nl);
        ensures nl != null ==> \result[0] == \unfolding [read]list_perm(nl) \in nl.node;
    pure static seq<Node> toSeq(NodeList nl) 
        = nl == null ? seq<Node> {}
                     : \unfolding [read]list_perm(nl) \in seq<Node>{nl.node} + toSeq(nl.next);

    /// turn list of Nodes into a sequence of ints, containing just the keys of the Nodes
        requires [read]list_perm(nl);
        ensures |\result| == size(nl);
        ensures nl != null ==> \result[0] == \unfolding [read]list_perm(nl) \in nl.node.key;
    pure static seq<int> toSeqKeys(NodeList nl) 
        = nl == null ? seq<int> {}
                     : \unfolding [read]list_perm(nl) 
                        \in seq<int>{nl.node.key} + toSeqKeys(nl.next);
    
            
    /// check whether given list is sorted (according to the Node keys)
        requires [read]list_perm(nl);
    pure static boolean sorted(NodeList nl) 
        = Util.sorted(toSeqKeys(nl)); 
    @*/


    /***********
     Actual Code
     ***********/

    /// Constructor, prepending given Node to given NodeList
    /*@
        requires node != null;
        requires Node.node_perm(node, write);
        requires list_perm(nl);
        requires sorted(nl) 
                 && (\let seq<int> nlKeys = toSeqKeys(nl); 
                        (\forall int i; 0<=i && i<|nlKeys|; node.key <= nlKeys[i]));
        ensures list_perm(this);
        ensures sorted(this);
        ensures size(this) == \old(size(nl)) + 1;
        ensures toSeqKeys(this) == \old(seq<int>{node.key} + toSeqKeys(nl));
        ensures \unfolding list_perm(this) 
                    \in this.node == node
                        && this.next==nl
                        && toSeq(this.next) == \old(toSeq(nl));
    @*/
    public NodeList(Node node, NodeList nl) {
        this.node = node; 
        this.next = nl;
        /*@ fold list_perm(this); @*/
    }
    
    
    /// number of Nodes in this list
    /*@
        given frac p;
        context [p]list_perm(this);
        ensures \result == size(this);
    @*/
    public int length() {
        int len = 1;
        /*@ unfold [p]list_perm(this); @*/
        if (this.next != null) {
            len += this.next.length() /*@ with {p=p;} @*/;
        }
        /*@ fold [p]list_perm(this); @*/
        return len;
    }
    


    /// add given Node at end of this list
    /*@
        requires node != null;
        requires Node.node_perm(node, write);
        requires list_perm(this);
        requires sorted(this) 
                 && (\forall int i; 0<=i && i<|toSeqKeys(this)|; 
                        toSeqKeys(this)[i] <= node.key);
        ensures list_perm(this);
        ensures size(this) == 1 + \old(size(this));
        ensures toSeq(this) == \old(toSeq(this) + seq<Node>{node});
        ensures toSeqKeys(this) == \old(toSeqKeys(this) + seq<int>{node.key});
        ensures sorted(this);
    @*/
    public void append(Node node) {
        /*@ ghost seq<int> keys = toSeqKeys(this);  @*/
        /*@ unfold list_perm(this); @*/
        /*@ assert toSeqKeys(this.next) == tail(keys); @*/
        this.next = appendRec(node, this.next);
        /*@ fold list_perm(this); @*/
    }
    
    /// recursive method to add given Node at end of given list
    /*@ 
        requires list_perm(nl);
        requires node != null;
        requires Node.node_perm(node, 1);
        requires sorted(nl) 
                 && (\forall int i; 0<=i && i<|toSeqKeys(nl)|; 
                        toSeqKeys(nl)[i] <= node.key);
        ensures list_perm(\result);
        ensures size(\result) == 1 + \old(size(nl));
        ensures toSeq(\result) == \old(toSeq(nl) + seq<Node>{node});
        ensures toSeqKeys(\result) == \old(toSeqKeys(nl) + seq<int>{node.key});
        ensures sorted(\result);
    @*/
    static NodeList appendRec(Node node, NodeList nl) {
        if (nl == null) {
            /*@ fold list_perm(null); @*/
            return new NodeList(node, null);
        } else {
            /*@ ghost seq<int> keys = toSeqKeys(nl);  @*/
            /*@ unfold list_perm(nl); @*/
            /*@ assert toSeqKeys(nl.next) == tail(keys); @*/
            nl.next = appendRec(node, nl.next);
            /*@ fold list_perm(nl); @*/
        }
        return nl;
    }
    
    

    /// append an entire NodeList at the end of the given list
    /*@
        requires list_perm(front) ** list_perm(back);
        requires sorted(front) && sorted(back)
                 && (\forall int i; 0<=i && i<|toSeqKeys(front)|; 
                        (\forall int j; 0<=j && j<|toSeqKeys(back)|; 
                            toSeqKeys(front)[i] <= toSeqKeys(back)[j]));
        ensures list_perm(\result);
        ensures size(\result) == \old(size(front)) + \old(size(back));
        ensures toSeq(\result) == \old(toSeq(front)) + \old(toSeq(back));
        ensures toSeqKeys(\result) == \old(toSeqKeys(front)) + \old(toSeqKeys(back));
        ensures sorted(\result);
    @*/
    public static NodeList extend(NodeList front, NodeList back) {
        if (front == null) {
            return back;
        } else {
            /*@ 
            ghost seq<int> frontKeys = toSeqKeys(front); 
            unfold list_perm(front); 
            assert toSeqKeys(front.next) == tail(frontKeys);
            @*/
            front.next = extend(front.next, back);
            /*@ fold list_perm(front); @*/
            return front;
        }
    }
    
    
    
    /// turn an RB tree into a NodeList
    /*@
        requires Tree.tree_perm(node);
        requires Tree.noDBlack(node);
        requires Tree.sortedKeySeq(node);
        ensures node != null ==> \result != null;
        ensures list_perm(\result);
        ensures \old(Tree.toSeq(node)) == toSeq(\result);
        ensures \old(Tree.toSeqKeys(node)) == toSeqKeys(\result);
        ensures sorted(\result);
    @*/
    public static NodeList fromTree(Node node) {
        if (node == null) {
            /*@ fold list_perm(null); @*/
            return null;
        }
        
        /*@ unfold Tree.tree_perm(node); @*/
        NodeList left = fromTree(node.left);
        NodeList right = fromTree(node.right);
        /*@ assume Perm(Integer.MIN_VALUE, read); @*/
        NodeList res = appendRec(node, left);
        res = extend(res, right);
        return res;
    }
    
}
