/**
 * FIFO queue using a ListList (list of list of nodes). Uses producer-consumer pattern.
 * @author Lukas Armborst, University of Twente
 * @year 2021
 */

final class ListListQueue {

    /******
     Fields
     ******/
    /// current batch being iterated by consumer
    NodeListIterator curNLIterator;
    /// list of complete batches still to be processed by consumer
    ListList batches;
    /// current batch being filled by producer
    NodeList filling;
    /// consumer has read all batches (the one in curNLIterator might still be in progress)
    boolean isLastBatch;
    /// producer has sent it all
    boolean finalised;
    
    /*@ 
    /// sequence representation of the entire ListList (incl. values already read). producer
    ///   and consumer each have their own, with the consumer's being a prefix of the producer's
    ghost seq<Node> allP;
    ghost seq<Node> allC;
    /// similar sequence representation of all the keys of the nodes in the ListList
    ghost seq<int> keysP;
    ghost seq<int> keysC;
    /// index in above seq from which consumer will read next; start point of curNLIterator
    ghost int readHead;
    /// start index in above seq for the batches; end point of curNLIterator
    ghost int batchHead;
    /// shadow of finalised flag, accessible to consumer
    ghost boolean finalisedC;
    /// largest key of all nodes in the ListList so far, new keys must be larger
    ghost int maxKey;
    /// largest key of all nodes in the batches so far, all keys in filling must be larger
    ghost int maxKeyBatches;
    @*/


    /// when working in batches, this is the maximal size of a batch
    /*@
        ensures \result == 10;
    @*/
    public /*@ pure @*/ static int maxLength() {
        return 10;
    }


    /*********
     Resources
     *********/
    /*@
    /// resource to group access rights of producer, as well as invariant properties
    public resource producer() 
        = Perm(filling, write) ** NodeList.list_perm(filling) 
          ** Perm(allP, 1\2)
          ** Perm(keysP, 1\2)
          ** Perm(finalised, 1\2)
          ** Perm(maxKey, write)
          ** Perm(maxKeyBatches, 1\2)
          ** NodeList.size(filling) < maxLength()
          ** (finalised ==> NodeList.size(filling)==0)
          ** |allP| == |keysP|
          ** NodeList.sorted(filling)
          ** Util.sorted(keysP)
          ** maxKeyBatches <= maxKey
          ** (\let seq<int> fillKeys = NodeList.toSeqKeys(filling); 
                (\forall int i; 0<=i && i<|fillKeys|; 
                    maxKeyBatches <= fillKeys[i] 
                    && fillKeys[i] <= maxKey))
          ** (\forall int i; 0<=i && i<|keysP|; 
                keysP[i] <= maxKeyBatches)
          ;
    
    /// resource to group access rights of consumer, as well as invariant properties
    public resource consumer() 
        = Perm(allC, 1\2) ** Perm(finalisedC, 1\2)
          ** Perm(keysC, 1\2) 
          ** Perm(readHead, write) ** Perm(batchHead, 1\2) 
          ** Perm(isLastBatch, write) ** Perm(curNLIterator, write)
          ** (curNLIterator != null ==> curNLIterator.consumer() ** curNLIterator.current != null)
          ** Util.sorted(keysC)
          ** |allC| == |keysC|
          ** readHead>=0 ** readHead<=|allC| 
          ** batchHead <= |allC|
          ** (isLastBatch == (batchHead==|allC| && finalisedC))
          ** batchHead-readHead == |NodeListIterator.toSeq(curNLIterator)|
          ** Util.isInfix(NodeListIterator.toSeq(curNLIterator), allC, readHead)
          ** Util.isInfix(NodeListIterator.toSeqKeys(curNLIterator), keysC, readHead)
          ;
    
    /// shared access right stored in a lock, as well as invariant properties
    public resource lock_invariant() 
        = Perm(batches, write) ** ListList.list_perm(batches)
          ** Perm(allP, 1\2) ** Perm(allC, 1\2) 
          ** Perm(keysP, 1\2) ** Perm(keysC, 1\2) 
          ** Perm(finalised, 1\2) ** Perm(finalisedC, 1\2)
          ** Perm(batchHead, 1\2) 
          ** Perm(maxKeyBatches, 1\2)
          ** batchHead >= 0 ** batchHead == |allP| - |ListList.toSeq(batches)| 
          ** Util.isPrefix(allC, allP)
          ** Util.isPrefix(keysC, keysP)
          ** Util.sorted(keysP)
          ** Util.sorted(keysC)
          ** |allC| == |keysC|
          ** |allP| == |keysP|
          ** Util.isInfix(ListList.toSeq(batches), allP, batchHead)
          ** Util.isInfix(ListList.toSeqKeys(batches), keysP, batchHead)
          ** (finalisedC ==> finalised && allC == allP && keysC == keysP)
          ** ListList.sorted(batches) 
          ;
    @*/


    /****************
     Helper Functions
     ****************/

    /*@
    /// check whether the consumer has read all Nodes
        requires consumer();
    public pure inline boolean done()
        = \unfolding consumer() \in isLastBatch && curNLIterator==null;
    @*/
    
    /*@
    /// turn this into a bag, containing all Nodes that were ever in this list
        requires Perm(allP, read);
    public pure inline bag<Node> toBag() 
        = Util.toBag(allP);

    /// turn producer's view of this into a bag, containing all Nodes ever added to this list
        requires producer();
        ensures \unfolding producer() \in |\result| == |allP + NodeList.toSeq(filling)|;
    public pure bag<Node> toBagP() 
        = \unfolding producer() \in Util.toBag(allP + NodeList.toSeq(filling));

    /// turn consumer's view of this into a bag, containing all Nodes ever read from this list
        requires consumer();
        ensures \unfolding consumer() \in |\result| == readHead;
    public pure bag<Node> toBagC() 
        = \unfolding consumer() \in Util.toBag(allC, readHead);
    @*/
    
    
    
    /************
     Constructors
     ************/
    
    /// initialise an empty ListList, for the producer to fill
    /*@
        ensures producer() ** consumer() ** Perm(Integer.MIN_VALUE, read);
        ensures \unfolding producer() \in filling == null ** |allP| == 0 
                        ** !finalised ** maxKey == Integer.MIN_VALUE;
        ensures \unfolding consumer() \in curNLIterator == null ** |allC| == 0
                        ** !finalisedC ** readHead == 0 ** !isLastBatch;
        ensures toBagC() == bag<Node>{} ** toBagP() == bag<Node>{};
    @*/
    ListListQueue() {
        curNLIterator = null;
        batches = null;
        filling = null;
        finalised = false;
        isLastBatch = false;
        /*@ 
        ghost allP = seq<Node>{};
        ghost allC = seq<Node>{};
        ghost keysP = seq<int>{};
        ghost keysC = seq<int>{};
        ghost readHead = 0;
        ghost batchHead = 0;
        ghost finalisedC = false; 
        assume Perm(Integer.MIN_VALUE, read);
        ghost maxKey = Integer.MIN_VALUE;
        ghost maxKeyBatches = maxKey;

        fold consumer();
        fold NodeList.list_perm(filling);
        fold producer();
        fold ListList.list_perm(batches);
        fold lock_invariant(); 
        @*/
    }
    
    /// initialise a ListList containing a single NodeList (not allowing further production)
    /*@
        requires it != null ** it.consumer() ** it.current != null;
        ensures consumer() ** producer();
        ensures \unfolding producer() \in finalised ** allP == \old(NodeListIterator.toSeq(it)) 
                                            ** keysP == \old(NodeListIterator.toSeqKeys(it));
        ensures \unfolding consumer() \in readHead == 0 ** curNLIterator == it ** isLastBatch;
        ensures toBagC() == bag<Node>{};
    @*/
    ListListQueue(NodeListIterator it) {
        curNLIterator = it;
        batches = null;
        filling = null;
        finalised = true;
        isLastBatch = true;
        /*@ 
        ghost allP = NodeListIterator.toSeq(it);
        ghost allC = allP;
        ghost keysP = NodeListIterator.toSeqKeys(it);
        ghost keysC = keysP;
        ghost readHead = 0;
        ghost batchHead = |allC|;
        ghost finalisedC = true; 
        ghost maxKey = keysC[|keysC|-1];
        ghost maxKeyBatches = maxKey;

        fold consumer();
        fold NodeList.list_perm(filling);
        fold producer();
        fold ListList.list_perm(batches);
        fold lock_invariant(); 
        @*/
    }
    


    /******************
     Producer's methods
     ******************/
    
    /// producer adding given Node to end of ListList
    /*@
        context producer();
        context \unfolding producer() \in !finalised;
        requires node != null;
        requires Node.node_perm(node, write);
        requires \unfolding producer() \in node.key >= maxKey;
        ensures \unfolding producer() \in maxKey == \old(node.key);
        ensures \unfolding producer() \in allP + NodeList.toSeq(filling) 
                    == \old(\unfolding producer() \in allP + NodeList.toSeq(filling)) 
                        + seq<Node>{node};
        ensures toBagP() == \old(toBagP()) + bag<Node>{node};
        ensures \unfolding producer() \in keysP + NodeList.toSeqKeys(filling) 
                    == \old(\unfolding producer() \in keysP + NodeList.toSeqKeys(filling) 
                            + seq<int>{node.key});
    @*/
    public void append(Node node) {
        /*@ unfold producer(); @*/
        /*@ ghost maxKey = node.key; @*/
        
        /// add node to "filling" batch
        if (filling == null) {
            /*@ fold NodeList.list_perm(null); @*/
            filling = new NodeList(node, null);
        } else {
            filling.append(node);
        }
        
        /*@ 
        /// helper assertions to prove post-condition
        assert allP + NodeList.toSeq(filling) 
                    == \old(\unfolding producer() \in allP + NodeList.toSeq(filling)) 
                        + seq<Node>{node};
        assert Util.isPrefix(\old(\unfolding producer() \in allP + NodeList.toSeq(filling)), 
                            allP + NodeList.toSeq(filling));
        /// turning the first n elements into a bag is the same, 
        ///     even if we extend the underlying seq beyond n
        assert Util.toBagPrefixLemma(
                            \old(\unfolding producer() \in allP + NodeList.toSeq(filling)), 
                            allP + NodeList.toSeq(filling),
                            0, 
                            |\old(\unfolding producer() \in allP + NodeList.toSeq(filling))|, 
                            \old(toBagP()));
        /// turning a longer prefix into a bag gives the old bag + the respective seq element
        assert Util.toBagMaxIdxIncrLemma(allP + NodeList.toSeq(filling),
                            0, 
                            |\old(\unfolding producer() \in allP + NodeList.toSeq(filling))|, 
                            \old(toBagP())); 
        @*/
        
        /// if filling is full, add it to the list of batches
        if (filling.length() /*@ with {p=1\2;} @*/ >= maxLength()) {
            submitFilling();
        }
        /*@ fold producer(); @*/
    }
    
    /// producer adding a completed batch to ListList for consumption
    /*@
        context Perm(finalised, 1\2) 
                ** Perm(filling, write) ** Perm(allP, 1\2) ** Perm(keysP, 1\2) 
                ** Perm(maxKey, 1\2) ** Perm(maxKeyBatches, 1\2);
        context NodeList.list_perm(filling);
        context maxKeyBatches <= maxKey;
        context !finalised;
        context (\forall int i; 0<=i && i<|keysP|; 
                    keysP[i] <= maxKeyBatches);
        requires filling != null;
        requires Util.sorted(keysP) ** NodeList.sorted(filling);
        requires (\let seq<int> newKeys = NodeList.toSeqKeys(filling);
                    (\forall int i; 0<=i && i<|newKeys|; 
                        maxKeyBatches <= newKeys[i] 
                        && newKeys[i] <= maxKey));
        ensures filling == null;
        ensures allP == \old(allP) + \old(NodeList.toSeq(filling));
        ensures keysP == \old(keysP) + \old(NodeList.toSeqKeys(filling));
        ensures Util.toBag(allP + NodeList.toSeq(filling)) 
                == \old(Util.toBag(allP + NodeList.toSeq(filling)));
        ensures Util.sorted(keysP);
    @*/
    synchronized void submitFilling() {
        /// workaround for synchronized
        /*@ assume lock_invariant(); @*/
        
        /*@ unfold lock_invariant(); @*/
        /*@ 
        ghost seq<Node> newSeq = NodeList.toSeq(filling);
        ghost allP = allP + newSeq; 
        ghost keysP = keysP + NodeList.toSeqKeys(filling); 
        ghost maxKeyBatches = maxKey; 
        @*/
        /// a prefix of X is also a prefix of any extension of X
        /*@ assert Util.prefixExtensionLemma(allC, \old(allP), newSeq, allP); @*/
        /// learn that size of toSeq(batches) is size(batches)
        /*@ assert ListList.seqSizeLemma(batches); @*/
        if (batches == null) {
            batches = new ListList(filling);
        } else {
            batches.append(filling);
        }
        /*@ assert !finalised; @*/
        
        filling = null;
        /*@ fold NodeList.list_perm(filling); @*/
        
        /*@ fold lock_invariant(); @*/
    }
    
    /// producer marking ListList as finished, no more Nodes to be added
    /*@
        context producer();
        requires \unfolding producer() \in !finalised;
        ensures \unfolding producer() \in finalised;
        ensures \unfolding producer() \in allP == \old(\unfolding producer() \in allP 
                                                            + NodeList.toSeq(filling));
        ensures \unfolding producer() \in keysP == \old(\unfolding producer() \in keysP 
                                                            + NodeList.toSeqKeys(filling));
        ensures toBagP() == \old(toBagP());
    @*/
    public void finalise() {
        /*@ unfold producer(); @*/
        if (filling != null) {
            submitFilling();
        }
        /// synchronized
        {
            /*@ assume lock_invariant(); @*/
            /*@ unfold lock_invariant(); @*/
            finalised = true;
            /*@ fold lock_invariant(); @*/
        }
        /*@ fold producer(); @*/
    }
    
    
    /******************
     Consumer's methods
     ******************/
    
    /// check whether there are more Nodes to be read
    /*@
        context consumer();
        ensures \unfolding consumer() \in readHead == \old(\unfolding consumer() \in readHead);
        ensures Util.isPrefix(\old(\unfolding consumer() \in allC), 
                              \unfolding consumer() \in allC);
        ensures Util.isPrefix(\old(\unfolding consumer() \in keysC), 
                              \unfolding consumer() \in keysC);
        ensures toBagC() == \old(toBagC());
        ensures \result ==> \unfolding consumer() \in readHead<|allC| && readHead<|keysC|;
        ensures !\result ==> done();
    @*/
    public boolean hasNext() {
        /*@ unfold consumer(); @*/
        if (curNLIterator == null) {
            if (isLastBatch) {
                /*@ fold consumer(); @*/
                return false;
            }
            /*@ fold consumer(); @*/
            getBatch();
            /*@ unfold consumer(); @*/
        }
        boolean res = curNLIterator != null;
        /*@ fold consumer(); @*/
        return res;
    }
    
    /// read a Node from the ListList
    /*@
        context consumer(); 
        ensures \old(\unfolding consumer() \in finalisedC) 
                ==> \unfolding consumer() 
                        \in finalisedC
                            && \old(\unfolding consumer() \in |allC|) == |allC|;
        ensures Util.isPrefix(\old(\unfolding consumer() \in allC), 
                              \unfolding consumer() \in allC);
        ensures Util.isPrefix(\old(\unfolding consumer() \in keysC), 
                              \unfolding consumer() \in keysC);
        ensures \result != null ==> Node.node_perm(\result, write);
        ensures \result != null 
                ==> \unfolding consumer() 
                        \in readHead == \old(\unfolding consumer() \in readHead) + 1;
        ensures \result != null 
                ==> \result == (\let int idx = \old(\unfolding consumer() \in readHead);
                                  \unfolding consumer() \in allC[idx]);
        ensures \result != null 
                ==> \result.key == (\let int idx = \old(\unfolding consumer() \in readHead);
                                      \unfolding consumer() \in keysC[idx]);
        ensures \result != null ==> toBagC() == \old(toBagC()) + bag<Node>{\result};
        ensures \result == null ==> done() 
                                    && \unfolding consumer() 
                                        \in readHead == \old(\unfolding consumer() \in readHead)
                                            && |allC| == readHead; 
        ensures \result == null ==> toBagC() == \old(toBagC());
    @*/
    public Node getNext() {
        /*@ unfold consumer(); @*/
        if (curNLIterator == null) {
            if (isLastBatch) {
                /*@ assert allC == \old(\unfolding consumer() \in allC); @*/
                /*@ assert Util.isPrefix(\old(\unfolding consumer() \in allC), allC); @*/
                /*@ fold consumer(); @*/
                return null;
            }
            /*@ fold consumer(); @*/
            if (!getBatch()) {
                /*@ unfold consumer(); @*/
                /*@ assert allC == \old(\unfolding consumer() \in allC); @*/
                /*@ assert Util.isPrefix(\old(\unfolding consumer() \in allC), allC); @*/
                /*@ fold consumer(); @*/
                return null;
            }
            /*@ unfold consumer(); @*/
        }
        /// tail of an infix is an infix too, at index+1
        /*@ assert Util.infixTailLemma(NodeListIterator.toSeq(curNLIterator), allC, readHead); @*/
        Node res = curNLIterator.getNext();
        /// turning larger infix into bag gives previous infix bag + respective additional Node
        /*@ assert Util.toBagMaxIdxIncrLemma(allC, 0, readHead, \old(toBagC())); @*/
        /*@ ghost readHead = readHead + 1; @*/
        if (!curNLIterator.hasNext() /*@ with {p=1\2;} @*/) {
            curNLIterator = null;
        }
        /*@ fold consumer(); @*/
        return res;
    }
    
    /// load a batch from the shared ListList into the consumer's own memory for easy access
    /// return whether a batch could be loaded
    /*@
        context consumer();
        requires \unfolding consumer()
                    \in curNLIterator==null && readHead==batchHead && !isLastBatch;
        ensures \old(\unfolding consumer() \in finalisedC) 
                ==> \unfolding consumer() 
                        \in finalisedC
                            && \old(\unfolding consumer() \in |allC|) == |allC|;
        ensures Util.isPrefix(\old(\unfolding consumer() \in allC), 
                                  \unfolding consumer() \in allC);
        ensures Util.isPrefix(\old(\unfolding consumer() \in keysC), 
                                  \unfolding consumer() \in keysC);
        ensures \unfolding consumer() \in readHead == \old(\unfolding consumer() \in readHead);
        ensures \unfolding consumer() \in batchHead == \old(\unfolding consumer() \in batchHead) 
                                                       + |NodeListIterator.toSeq(curNLIterator)|;
        ensures toBagC() == \old(toBagC());
        ensures \result ==> \unfolding consumer() \in curNLIterator!=null && readHead<|allC|;
        ensures !\result ==> done();
    @*/
    boolean getBatch() {
        /// check if there are more batches to read
        /// synchronized
        {
            /*@ assume lock_invariant(); @*/
            /*@ unfold lock_invariant(); @*/
            if (finalised && batches == null) {
                /*@ unfold consumer(); @*/
                isLastBatch = true;
                /*@ ghost finalisedC = finalised; @*/
                /*@ fold consumer(); @*/
                /*@ fold lock_invariant(); @*/
                return false;
            }
            /*@ fold lock_invariant(); @*/
        }

        /// wait for a batch to be available, or the producer to signal that there are no more
        NodeList head = null;
        /*@ fold NodeList.list_perm(head); @*/
        boolean waiting = true;
        /*@
            loop_invariant consumer(); 
            loop_invariant Util.isPrefix(\old(\unfolding consumer() \in allC), 
                                         \unfolding consumer() \in allC);
            loop_invariant Util.isPrefix(\old(\unfolding consumer() \in keysC), 
                                         \unfolding consumer() \in keysC);
            loop_invariant \old(\unfolding consumer() \in finalisedC) 
                           ==> \unfolding consumer() 
                                  \in finalisedC
                                      && \old(\unfolding consumer() \in |allC|) == |allC|;
            loop_invariant toBagC() == \old(toBagC());
            loop_invariant waiting ==> head==null;
            loop_invariant waiting ==> \unfolding consumer() \in readHead==batchHead;
            loop_invariant waiting ==> \unfolding consumer() \in curNLIterator==null;
            loop_invariant waiting ==> \unfolding consumer() \in !isLastBatch;
            loop_invariant !waiting && head!=null
                            ==> (\unfolding consumer() \in curNLIterator!=null && readHead<|allC|);
            loop_invariant !waiting && head==null ==> done();
            loop_invariant \unfolding consumer() 
                              \in waiting ? batchHead == \old(\unfolding consumer() \in batchHead)
                                          : batchHead == \old(\unfolding consumer() \in batchHead) 
                                                         + |NodeListIterator.toSeq(curNLIterator)|;
        @*/
        while (waiting) {
            /// synchronized(this) 
            {
                /// workaround for synchronized
                /*@ assume lock_invariant(); @*/
                /*@ unfold lock_invariant(); @*/
                /*@ unfold consumer(); @*/
                if (batches != null) {
                    /// new batch available
                    
                    /*@ 
                    /// save current state as reference for later proof
                    ghost seq<Node> oldSeq = ListList.toSeq(batches); 
                    ghost seq<int> batchKeys = ListList.toSeqKeys(batches); 
                    unfold ListList.list_perm(batches); 
                    /// if seq is sorted, so are it's constituents
                    assert Util.sortedLemma(batchKeys, 
                                            NodeList.toSeqKeys(batches.nodeList), 
                                            ListList.toSeqKeys(batches.next), 
                                            seq<int>{}); 
                    @*/
                    
                    head = batches.nodeList;
                    batches = batches.next;
                    waiting = false;
                    
                    /// prove that new "batches" still satifies invariants
                    /*@ 
                    assert oldSeq == NodeList.toSeq(head) + ListList.toSeq(batches);
                    assert batchKeys == NodeList.toSeqKeys(head) 
                                        + ListList.toSeqKeys(batches);
                    assert (\forall int i; 0<=i && i<|ListList.toSeqKeys(batches)|;
                                ListList.toSeqKeys(batches)[i] 
                                    == batchKeys[i+|NodeList.toSeqKeys(head)|]);
                    /// new "batches" is still a suffix of allP
                    assert Util.suffixShortenLemma(oldSeq, 
                                                   allP, 
                                                   NodeList.toSeq(head), 
                                                   ListList.toSeq(batches));
                    /// keys in new "batches" form still an infix of keysP
                    assert Util.infixSuffixLemma(batchKeys, 
                                                 keysP, 
                                                 NodeList.toSeqKeys(head), 
                                                 ListList.toSeqKeys(batches),
                                                 batchHead); 
                    @*/
                    /*@
                    ghost batchHead = batchHead + |NodeList.toSeq(head)|; 
                    @*/
                    curNLIterator = new NodeListIterator(head);
                    /// turning n entries of prefix into bag is same as using n entries of whole seq
                    /*@ assert Util.toBagPrefixLemma(allC, allP, 0, readHead, \old(toBagC())); @*/
                    /*@ ghost allC = allP; @*/
                    /*@ ghost keysC = keysP; @*/
                    /*@ ghost finalisedC = finalised; @*/
                } 
                /// check if there are more batches to read
                if (finalised && batches == null) {
                    waiting = false;
                    isLastBatch = true;
                    /// turning n entries of prefix into bag is same as using n entries of whole seq
                    /*@ assert Util.toBagPrefixLemma(allC, allP, 0, readHead, \old(toBagC())); @*/
                    /*@ ghost allC = allP; @*/
                    /*@ ghost keysC = keysP; @*/
                    /*@ ghost finalisedC = finalised; @*/
                }
                /*@ fold consumer(); @*/
                /*@ fold lock_invariant(); @*/
            }
            if (waiting) {
                /// wait
            }
        }
        /*@ unfold consumer(); @*/
        boolean res = (curNLIterator!=null);
        /*@ fold consumer(); @*/
        return res;
    }
    
    
    /// Turn this ListList into an RB tree (if nothing has been consumed yet)
    /*@
        context consumer();
        requires \unfolding consumer() \in readHead == 0;
        ensures done();
        ensures Tree.tree_perm(\result);
        ensures Tree.toSeq(\result) == \unfolding consumer() \in allC;
        ensures Tree.validTree(\result);
    @*/
    public Node toTree() {
        /// wait for producer to finish generating the list
        boolean waiting = true;
        int numNodes = 0;
        /*@ loop_invariant consumer(); @*/
        /*@ loop_invariant \unfolding consumer() \in readHead == 0; @*/
        /*@ loop_invariant waiting ==> numNodes == 0; @*/
        /*@ loop_invariant !waiting ==> \unfolding consumer() \in numNodes == |allC|; @*/
        /*@ loop_invariant !waiting ==> \unfolding consumer() \in finalisedC; @*/
        while (waiting) {
            /// synchronized 
            {
                /*@ assume lock_invariant(); @*/
                /*@ unfold lock_invariant(); @*/
                if (finalised) {
                    waiting = false;
                    /*@ unfold consumer(); @*/
                    /*@ ghost allC = allP; @*/
                    /*@ ghost keysC = keysP; @*/
                    /*@ ghost finalisedC = finalised; @*/
                    if (batches != null) {
                        numNodes = batches.size() /*@ with {p=1\2;} @*/;
                    }
                    if (numNodes <= 0) {
                        isLastBatch = true;
                    }
                    if (curNLIterator != null) {
                        numNodes += curNLIterator.current.length() /*@ with {p=1\2;} @*/;
                    }
                    /// learn that size of toSeq(batches) is size(batches)
                    /*@ assert ListList.seqSizeLemma(batches); @*/
                    /* assert numNodes == |allC|; */
                    /*@ fold consumer(); @*/
                }
                /*@ fold lock_invariant(); @*/
            }
        }
        
        /// convert finalised list into tree
        Node res = null;
        /*@ fold Tree.tree_perm(res); @*/
        if (numNodes > 0) {
            int height = log2(numNodes);
            res = makeTree(height, numNodes);
            /*@ unfold Tree.tree_perm(res); @*/
            res.color = false;
            /*@ fold Tree.tree_perm(res); @*/
        }
        return res;
    }
    
    /// create an RB tree with given number of Nodes (and given height)
    /*@
        context consumer();
        context \unfolding consumer() \in finalisedC;
        requires height >= 0 && numNodes > 0;
        requires pow2(height)-2 < numNodes && numNodes < pow2(height+1);
        requires \unfolding consumer() \in numNodes<=(|allC|-readHead);
        ensures Tree.tree_perm(\result);
        ensures \result != null;
        ensures |Tree.toSeq(\result)| == numNodes;
        ensures \unfolding consumer() \in allC == \old(\unfolding consumer() \in allC);
        ensures \unfolding consumer() \in keysC == \old(\unfolding consumer() \in keysC);
        ensures \unfolding consumer() \in readHead == \old(\unfolding consumer() \in readHead) 
                                                        + numNodes;
        ensures \unfolding consumer() 
                    \in Util.isInfix(Tree.toSeq(\result), allC, 
                                         \old(\unfolding consumer() \in readHead));
        ensures \unfolding consumer() 
                    \in Util.isInfix(Tree.toSeqKeys(\result), keysC, 
                                         \old(\unfolding consumer() \in readHead));
        ensures \unfolding Tree.tree_perm(\result) \in \result.color == (height % 2 == 0);
        ensures Tree.blackHeight(\result) == (height+1)/2 + 1;
        ensures Tree.blackBalanced(\result);
        ensures Tree.noDBlack(\result) ** Tree.noDoubleRed(\result);
        ensures Tree.sortedKeySeq(\result);
    
    @*/
    Node makeTree(int height, int numNodes) {
        Node left = null;
        Node right = null;
        /*@ fold Tree.tree_perm(left); @*/
        /*@ fold Tree.tree_perm(right); @*/
        /*@ 
        unfold consumer();
        ghost int startIdx = readHead; 
        fold consumer();
        @*/
        
        if (numNodes > 1) {
            left = makeTree(height-1, numNodes/2);
        }
        /*@ ghost seq<Node> leftSeq = Tree.toSeq(left); @*/
        /*@ ghost seq<int> leftKeys = Tree.toSeqKeys(left); @*/
        
        Node res = getNext();
        /*@ assert \unfolding consumer() 
                    \in Util.infixAdditionLemma(leftSeq, seq<Node>{res}, allC, 
                                         startIdx); @*/
        /*@ assert res.key == \unfolding consumer() 
                                      \in keysC[startIdx+|leftKeys|]; @*/
        /*@ assert \unfolding consumer() 
                    \in Util.infixAdditionLemma(leftKeys, seq<int>{res.key}, keysC, 
                                         startIdx); @*/
        
        if (numNodes > 2) {
            int newNum = numNodes - numNodes/2 - 1;
            right = makeTree(height-1, newNum);
        }
        /*@ assert \unfolding consumer() 
                    \in Util.infixAdditionLemma(leftSeq+seq<Node>{res}, Tree.toSeq(right), allC, 
                                         startIdx); @*/
        /*@ assert \unfolding consumer() 
                    \in Util.infixAdditionLemma(leftKeys+seq<int>{res.key}, Tree.toSeqKeys(right), keysC, 
                                         startIdx); @*/
        
        res.left = left;
        res.right = right;
        res.color = (height % 2 == 0);
        res.dblack = false;
        res.dblackNull = false;
        
        /*@ fold Tree.tree_perm(res); @*/
        return res;
    }


    /************
     Math helpers
     ************/

    /*@
    /// compute 2 to the power of given value
        requires x >= 0;
        ensures \result >= 1;
        ensures x>0 ==> \result > 1 && pow2(x-1) == \result / 2;
        ensures \result > 1 ==> x>0;
    pure static int pow2(int x) 
        = x<=0 ? 1 : 2*pow2(x-1);
    @*/
    
    /// compute logarithm of given value (rounded down)
    /*@
        requires x > 0;
        ensures \result >= 0;
        ensures x>1 ==> \result>0 && log2(x/2)==\result-1;
        ensures pow2(\result) <= x && pow2(\result+1) >= x+1;
    @*/
    /*@ pure @*/ int log2(int x) { 
        if(x/2 > 0) {
            return log2(x/2)+1;
        } else {
            return 0; 
        }
    }

    /**************************
     Join Producer and Consumer
     **************************/

    /*@
    /// join permissions of producer and consumer back into one unit after they are done
        requires producer() ** consumer();
        requires \unfolding producer() \in finalised;
        requires done() == true;
        ensures Perm(allP, 1\2);
        ensures allP == \old(\unfolding producer() \in allP);
        ensures toBag() == \old(toBagP());
        ensures toBag() == \old(toBagC());
    ghost public synchronized void joinPC() {
        assume lock_invariant(); 
        unfold lock_invariant();

        unfold producer(); 
    }
    @*/

}
