/**
 * linked list of NodeLists, to facilitate processing in chunks
 * @author Lukas Armborst, University of Twente
 * @year 2021
 */

final class ListList {
    NodeList nodeList;
    ListList next;


    /*@
    /// recursive resource allowing access to all nodes in all chunks
    public static resource list_perm(ListList ll)
        = ll != null ==> Perm(ll.nodeList, write)
                        ** Perm(ll.next, write)
                        ** ll.nodeList != null ** NodeList.list_perm(ll.nodeList)
                        ** list_perm(ll.next);
    @*/
    
    
    /************************
     Sequence representations
    *************************/
    /*@
    /// turning a ListList into a sequence of Nodes
        requires [read]list_perm(ll);
        ensures ll == null ==> |\result| == 0;
        ensures ll != null ==> |\result| > 0;
    public static pure seq<Node> toSeq(ListList ll)
        = ll == null ? seq<Node> {}
                    : \unfolding [read]list_perm(ll)
                        \in NodeList.toSeq(ll.nodeList) + toSeq(ll.next);
    
    /// turning a ListList into a sequence of NodeLists (i.e. only unrolling top level)
        requires [read]list_perm(ll);
    static pure seq<NodeList> toNLSeq(ListList ll)
        = ll == null ? seq<NodeList> {}
                    : \unfolding [read]list_perm(ll)
                        \in seq<NodeList>{ll.nodeList} + toNLSeq(ll.next);
    
    /// turning ListList into sequence of ints, only containing the keys of the Nodes
        requires [read]list_perm(ll);
    public static pure seq<int> toSeqKeys(ListList ll)
        = ll == null ? seq<int> {}
                    : \unfolding [read]list_perm(ll) 
                        \in NodeList.toSeqKeys(ll.nodeList) + toSeqKeys(ll.next);


        
    /// tail of NL sequence is toNLSeq of ll.next
        requires [read]list_perm(ll);
        ensures ll != null 
                ==> tail(toNLSeq(ll)) == \unfolding [read]list_perm(ll) \in toNLSeq(ll.next);
        ensures \result;        
    static pure boolean toNLSeqTailLemma(ListList ll)
        = ll != null ==> toNLSeq(ll) == \unfolding [read]list_perm(ll) 
                                            \in seq<NodeList>{ll.nodeList} + toNLSeq(ll.next);

    /// toSeq and toSeqKeys create sequences of length size(ll)
        requires [read]list_perm(ll);
        ensures |toSeq(ll)| == size(ll);
        ensures |toSeqKeys(ll)| == size(ll);
        ensures \result;
    public static pure boolean seqSizeLemma(ListList ll)
        = ll != null ==> (\unfolding [read]list_perm(ll) \in seqSizeLemma(ll.next)) 
                         && (|toSeq(ll)| == \unfolding [read]list_perm(ll) 
                                              \in |NodeList.toSeq(ll.nodeList)| 
                                                  + |toSeq(ll.next)|)
                         && (|toSeqKeys(ll)| == \unfolding [read]list_perm(ll) 
                                                  \in |NodeList.toSeqKeys(ll.nodeList)| 
                                                      + |toSeqKeys(ll.next)|);
    @*/



    /*******************
     ListList properties
     *******************/
    /*@
    /// ListList is sorted if the keys are
        requires [read]list_perm(ll);
    public static pure boolean sorted(ListList ll)
        = Util.sorted(toSeqKeys(ll)); 


    /// total number of Nodes in given ListList
        requires [read]list_perm(ll);
    public static pure int size(ListList ll)
        = ll == null ? 0
            : \unfolding [read] list_perm(ll) \in NodeList.size(ll.nodeList) + size(ll.next);
    @*/



    /***********
     Actual Code
     ***********/

    /// Constructor for a ListList storing the given NodeList as its only element
    /*@ 
        requires nl != null;
        requires NodeList.list_perm(nl); 
        requires NodeList.sorted(nl);
        ensures list_perm(this);
        ensures \unfolding list_perm(this) \in this.nodeList==nl && this.next==null;
        ensures toSeq(this) == \old(NodeList.toSeq(nl));
        ensures toSeqKeys(this) == \old(NodeList.toSeqKeys(nl));
        ensures sorted(this);
    @*/
    public ListList(NodeList nl) {
        nodeList = nl;
        next = null;
        /*@ fold list_perm(null); @*/
        /*@ fold list_perm(this); @*/
    }
    
    
    /// total number of Nodes in this ListList
    /*@
        given frac p;
        context [p]list_perm(this);
        ensures \result == size(this);
    @*/
    public int size() {
        int size = 0;
        /*@ unfold [p]list_perm(this); @*/
        size = size + this.nodeList.length() /*@ with {p=p\2;} @*/;
        /*@ assert size == NodeList.size(this.nodeList); @*/
        if (this.next != null) {
            size = size + this.next.size() /*@ with {p=p\2;} @*/;
        }
        /*@ assert size == NodeList.size(this.nodeList) + size(this.next); @*/
        /*@ fold [p]list_perm(this); @*/
        return size;
    }
    
    
    /// add given NodeList to end of this ListList
    /*@
        context list_perm(this);
        requires nl != null;
        requires NodeList.list_perm(nl);
        requires sorted(this) ** NodeList.sorted(nl);
        requires Util.smallerSeq(toSeqKeys(this), NodeList.toSeqKeys(nl));
        ensures toSeq(this) == \old(toSeq(this) + NodeList.toSeq(nl));
        ensures toSeqKeys(this) == \old(toSeqKeys(this) + NodeList.toSeqKeys(nl));
        ensures sorted(this);
    @*/
    public void append(NodeList nl) 
    {
        /*@ assert toNLSeqTailLemma(this); @*/
        /*@ ghost seq<int> keys = toSeqKeys(this); @*/
        /*@ unfold list_perm(this); @*/
        /*@ assert Util.sortedLemma(keys, NodeList.toSeqKeys(this.nodeList),
                                    toSeqKeys(this.next), NodeList.toSeqKeys(nl)); @*/
        this.next = appendRec(nl, this.next);
        /*@ fold list_perm(this); @*/
    }
    
    
    /// recursive method to add given NodeList to end of given ListList
    /*@
        requires nl != null;
        requires list_perm(ll);
        requires NodeList.list_perm(nl);
        requires sorted(ll) ** NodeList.sorted(nl);
        requires Util.smallerSeq(toSeqKeys(ll), NodeList.toSeqKeys(nl));
        ensures list_perm(\result);
        ensures toSeq(\result) == \old(toSeq(ll) + NodeList.toSeq(nl));
        ensures toSeqKeys(\result) == \old(toSeqKeys(ll) + NodeList.toSeqKeys(nl));
        ensures sorted(\result);
    @*/
    ListList appendRec(NodeList nl, ListList ll) 
    {
        ListList res;
        if (ll == null) {
            res = new ListList(nl);
        } else {
            res = ll;
            /*@ assert toNLSeqTailLemma(res); @*/
            /*@ ghost seq<int> keys = toSeqKeys(res); @*/
            /*@ unfold list_perm(res); @*/
            /*@ assert Util.sortedLemma(keys, NodeList.toSeqKeys(res.nodeList),
                                        toSeqKeys(res.next), NodeList.toSeqKeys(nl)); @*/
            res.next = appendRec(nl, res.next);
            /*@ assert res.nodeList != null; @*/
            /*@ fold list_perm(res); @*/
        }
        return res;
    }

}
