package lib.utils;

import java.lang.reflect.*;
import java.util.*;

public class ParameterizedField {
	public final Field field;
	public final Class<?> parameterizedType;
	public final Class<?> parameterType;
	
	private ParameterizedField(Field field, Class<?> parameterizedType, Class<?> parameterType) {
		this.field = field;
		this.parameterizedType = parameterizedType;
		this.parameterType = parameterType;
	}
	
	public <T> T getValue(Class<T> targetClz) {
		try {
			return targetClz.cast(FieldUtils.getValue(field, parameterizedType));
		} catch (ReflectionException e) {
			throw new Error(e);
		}
	}
	
	@Override
	public String toString() {
		return "Field \"" + field.getName() + "\" of class \"" + field.getDeclaringClass().getCanonicalName() + "\"";
	}
	
	public static ParameterizedField obtain(Field f, Object... allowedClzList) throws ReflectionException {
		Set<Class<?>> allowedClasses = ClassUtils.makeSet(allowedClzList);
		Class<?> parameterizedType = null;
		
		for (Class<?> allowedClz : allowedClasses) {
			if (allowedClz.isAssignableFrom(f.getType())) {
				parameterizedType = f.getType();
				break;
			}
		}
		
		if (parameterizedType == null) {
			String msg = "Invalid field type \"" + f.getType().getCanonicalName() + "\"; should be one of the following:";
			
			for (Class<?> allowedClz : allowedClasses) {
				msg += "\n\t" + allowedClz.getCanonicalName();
			}
			
			throw new FieldReflectionException(f, msg);
		}
		
		Type otherFieldType = f.getGenericType();
		
		if (!(otherFieldType instanceof ParameterizedType)) {
			throw new FieldReflectionException(f, "Field should have a parameterized type!");
		}
		
		Type[] typeArgs = ((ParameterizedType) otherFieldType).getActualTypeArguments();
		
		if (typeArgs.length != 1) {
			throw new FieldReflectionException(f, "Field should have a type with 1 parameter!");
		}
		
		if (!(typeArgs[0] instanceof Class)) {
			throw new FieldReflectionException(f, "Field should have a class as parameter!");
		}
		
		return new ParameterizedField(f, parameterizedType, (Class<?>) typeArgs[0]);
	}
	
	@Override
	public int hashCode() {
		return Objects.hash(field, parameterType, parameterizedType);
	}
	
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (!(obj instanceof ParameterizedField)) {
			return false;
		}
		ParameterizedField other = (ParameterizedField) obj;
		return Objects.equals(field, other.field) && Objects.equals(toString(), other.toString()) && Objects.equals(parameterType, other.parameterType) && Objects.equals(parameterizedType, other.parameterizedType);
	}
}

