package lib.utils;

import java.lang.reflect.*;
import java.util.Collection;

public class FieldUtils {
	/**
	 * Determines if this is a (synthesized) field that contains the reference to an owner class.
	 * Probably.
	 */
	public static boolean isOwnerRef(Field f) {
		if (f.isSynthetic()) {
			if (!Modifier.isStatic(f.getDeclaringClass().getModifiers()) && f.getDeclaringClass().getDeclaringClass() != null) {
				if (f.getType() == f.getDeclaringClass().getDeclaringClass()) {
					return true;
				}
			}
		}
		
		return false;
	}
	
	/**
	 * A field is considered `standard' when it is public and final.
	 */
	public static void confirmStdField(Field f, ModReq.StaticReq staticReq) throws ReflectionException {
		int m = f.getModifiers();
		
		if (!Modifier.isPublic(m)) {
			throw new FieldReflectionException(f, "Field should be public!");
		}
		
		if (!Modifier.isFinal(m)) {
			throw new FieldReflectionException(f, "Field should be final!");
		}
		
		switch (staticReq) {
			case SHOULD_BE_STATIC:
				if (!Modifier.isStatic(m)) {
					throw new FieldReflectionException(f, "Field should be static!");
				}
				break;
			case SHOULD_NOT_BE_STATIC:
				if (Modifier.isStatic(m)) {
					throw new FieldReflectionException(f, "Field should NOT be static!");
				}
				break;
			case DONT_CARE:
				break;
		}
	}
	
	public static <T> T getValue(Object instance, Field f, Class<T> targetClz) throws ReflectionException {
		try {
			return targetClz.cast(f.get(instance));
		} catch (ClassCastException | SecurityException | IllegalAccessException | IllegalArgumentException e) {
			throw new FieldReflectionException(f, e);
		}
	}
	
	public static <T> T getValue(Field f, Class<T> targetClz) throws ReflectionException {
		if (Modifier.isStatic(f.getModifiers())) {
			return getValue(null, f, targetClz);
		}
		
		return getValue(ClassUtils.createStdInstance(f.getDeclaringClass()), f, targetClz);
	}
	
	@SuppressWarnings("unchecked")
	public static <T> T[] getValues(Object instance, Field f, Class<T> targetClz) throws ReflectionException {
		try {
			return (T[])f.get(instance);
		} catch (ClassCastException | SecurityException | IllegalAccessException | IllegalArgumentException e) {
			throw new FieldReflectionException(f, e);
		}
	}
	
	public static <T> T[] getValues(Field f, Class<T> targetClz) throws ReflectionException {
		if (Modifier.isStatic(f.getModifiers())) {
			return getValues(null, f, targetClz);
		}
		
		return getValues(ClassUtils.createStdInstance(f.getDeclaringClass()), f, targetClz);
	}
	
	/**
	 * All fields must have an allowed type.
	 */
	public static boolean hasAllowedType(Collection<Field> fields, Object... allowedTypeList) {
		for (Field f : fields) {
			if (!hasAllowedType(f, allowedTypeList)) {
				return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Field must have an allowed type.
	 */
	public static boolean hasAllowedType(Field f, Object... allowedTypeList) {
		for (Object allowedType : allowedTypeList) {
			if (((Class<?>)allowedType).isAssignableFrom(f.getType())) {
				return true;
			}
		}
		
		return false;
	}
	
	public static void confirmAssigned(Field f, ModReq.AssignedReq assignedReq) throws ReflectionException {
		switch (assignedReq) {
			case SHOULD_BE_ASSIGNED:
				if (f.getType().isArray()) {
					Object[] values = FieldUtils.getValues(f, Object.class);
					
					if (values == null) {
						throw new FieldReflectionException(f, "Field should be assigned!");
					}
					
					for (int index = 0; index < values.length; index++) {
						if (values[index] == null) {
							throw new FieldReflectionException(f, "Field element " + index + " + should be assigned!");
						}
					}
				} else {
					if (getValue(f, Object.class) == null) {
						throw new FieldReflectionException(f, "Field should be assigned!");
					}
				}
				break;
			case SHOULD_NOT_BE_ASSIGNED:
				if (f.getType().isArray()) {
					Object[] values = getValues(f, Object.class);
					
					if (values != null) {
						throw new FieldReflectionException(f, "Field should NOT be assigned!");
					}
				} else {
					if (getValue(f, Object.class) == null) {
						throw new FieldReflectionException(f, "Field should NOT be assigned!");
					}
				}
				break;
			case DONT_CARE:
				break;
		}
	}
}
