package lib.utils;

import java.util.*;

public class Copyables implements Composable {
	private List<Copyable> items;
	private Map<Copyable, Copyable> ownerPerItem;
	private Map<Copyable, Copyable> copyPerOriginal;
	
	public Copyables() {
		items = new ArrayList<Copyable>();
		ownerPerItem = new HashMap<Copyable, Copyable>();
		copyPerOriginal = new HashMap<Copyable, Copyable>();
	}
	
	public Copyables(Copyables source) {
		this();
		
		for (Copyable item : source.items) {
			Copyable itemCopy = item.createCopy(this);
			
			if (copyPerOriginal.containsKey(item)) {
				throw new Error("Copying " + item + " multiple times!");
			}
			
			if (!item.getClass().isInstance(itemCopy)) {
				throw new Error("Copying " + item + " to " + itemCopy + "!");
			}
			
			copyPerOriginal.put(item, itemCopy);
		}
		
		for (Copyable item : items) {
			item.resolveRefsAfterCopy();
		}
		
		updateOwners();
	}
	
	public <T extends Copyable> T resolve(T original, Class<T> clz) {
		if (original == null) {
			return null;
		}
		
		Copyable result = copyPerOriginal.get(original);
		
		if (result == null) {
			throw new Error("Resolving " + original + " to null!");
		}
		
		if (clz.isInstance(result)) {
			return clz.cast(result);
		}
		
		throw new Error("Resolving " + original + " to " + result + ", which is not of type " + clz.getCanonicalName() + "!");
	}
	
	public void updateOwners() {
		ownerPerItem.clear();
		
		for (Copyable item : items) {
			List<? extends Copyable> ownedItems = item.getOwnedCopyables();
			
			if (ownedItems != null) {
				for (Copyable ownedItem : ownedItems) {
					Copyable currentOwner = ownerPerItem.get(ownedItem);
					
					if (currentOwner != null) {
						throw new Error("Should not happen!");
					}
					
					ownerPerItem.put(ownedItem, item);
				}
			}
		}
		
		for (Copyable item : items) {
			item.updateOwner(this);
		}
	}
	
	public <T extends Copyable> T getOwner(Copyable ownedItem, Class<T> clz) {
		Copyable result = ownerPerItem.get(ownedItem);
		
		if (result == null) {
			return null;
		}
		
		if (clz.isInstance(result)) {
			return clz.cast(result);
		}
		
		throw new Error(ownedItem + " is owned by " + result + ", which is not of type " + clz.getCanonicalName() + "!");
	}
	
	public void addItem(Copyable item, Copyable owner) {
		if (items.contains(item)) {
			throw new Error("Adding " + item + " multiple times!");
		}
		
		items.add(item);
		ownerPerItem.put(item, owner);
		item.updateOwner(this);
	}
	
	@Override
	public Copyable getOwner() {
		return null;
	}
	
	@Override
	public List<? extends Copyable> getOwnedComponents() {
		return items;
	}
}
