package lib.utils;

import java.lang.reflect.*;
import java.util.*;
import java.util.function.Predicate;

import lib.blocks.common.Pulse;

public class ClassUtils {
	public static <T> T getFieldValue(Class<?> clz, String fieldName, Class<T> targetClz, T fallback) throws ReflectionException {
		try {
			Field f = clz.getField(fieldName);
			return FieldUtils.getValue(f, targetClz);
		} catch (NoSuchFieldException e) {
			return fallback;
		} catch (SecurityException e) {
			throw new ClassReflectionException(clz, e);
		}
	}
	
	public static Field getOwnerRef(Class<?> clz) throws ReflectionException {
		for (Field f : clz.getDeclaredFields()) {
			if (FieldUtils.isOwnerRef(f)) {
				return f;
			}
		}
		
		throw new ClassReflectionException(clz, "Found no owner reference!");
	}
	
	/**
	 * All fields of a class, including inherited, but NOT including owner references (if they exist).
	 */
	public static Set<Field> getAllFields(Class<?> clz) {
		Set<Field> result = new HashSet<Field>();
		
		for (Field f : clz.getFields()) {
			if (!FieldUtils.isOwnerRef(f)) {
				result.add(f);
			}
		}
		
		return result;
	}
	
	public static Set<Field> getDeclaredFields(Class<?> clz) {
		Set<Field> result = new HashSet<Field>();
		
		for (Field f : clz.getDeclaredFields()) {
			if (!FieldUtils.isOwnerRef(f)) {
				result.add(f);
			}
		}
		
		return result;
	}
	
	public static Set<Class<?>> makeSet(Object... classes) {
		Set<Class<?>> result = new HashSet<Class<?>>();
		
		for (Object clz : classes) {
			if (clz instanceof Class) {
				result.add((Class<?>)clz);
			} else {
				throw new Error(clz.getClass().getCanonicalName() + " is not a class!");
			}
		}
		
		return result;
	}
	
	public static Set<Class<?>> getDeclaredClasses(Class<?> clz) {
		Set<Class<?>> result = new HashSet<Class<?>>();
		
		for (Class<?> declaredClz : clz.getDeclaredClasses()) {
			result.add(declaredClz);
		}
		
		return result;
	}
	
	public static void confirmUnnested(Class<?> clz) throws ReflectionException {
		if (clz.getDeclaringClass() != null) {
			throw new ClassReflectionException(clz, "Class should not be nested in class \"" + clz.getDeclaringClass().getCanonicalName() + "\"!");
		}
	}
	
	public static void confirmStdClass(Class<?> clz, ModReq.StaticReq staticReq) throws ReflectionException {
		int m = clz.getModifiers();
		
		switch (staticReq) {
			case SHOULD_BE_STATIC:
				if (!Modifier.isStatic(m)) {
					throw new ClassReflectionException(clz, "Class should be static!");
				}
				break;
			case SHOULD_NOT_BE_STATIC:
				if (Modifier.isStatic(m)) {
					throw new ClassReflectionException(clz, "Class should NOT be static!");
				}
				break;
			case DONT_CARE:
				break;
		}
		
		if (!Modifier.isPublic(m)) {
			throw new ClassReflectionException(clz, "Class should be public!");
		}
		
		if (Modifier.isAbstract(m)) {
			throw new ClassReflectionException(clz, "Class should NOT be abstract!");
		}
		
		if (clz.getTypeParameters().length > 0) {
			throw new ClassReflectionException(clz, "Class should NOT have type parameters!");
		}
		
		for (Constructor<?> c : clz.getDeclaredConstructors()) {
			switch (c.getParameterCount()) {
				case 0:
					break;
				case 1:
					if (!Modifier.isStatic(m) && clz.getDeclaringClass() != null) {
						if (c.getParameterTypes()[0] != clz.getDeclaringClass()) {
							throw new ClassReflectionException(clz, "Class should only have parameterless constructors!");
						}
					} else {
						throw new ClassReflectionException(clz, "Class should only have parameterless constructors!");
					}
					break;
				default:
					throw new ClassReflectionException(clz, "Class should only have parameterless constructors!");
			}
			
			if (c.getTypeParameters().length > 0) {
				throw new ClassReflectionException(clz, "Class should only have constructors WITHOUT type parameters!");
			}
			
			if (!Modifier.isPublic(c.getModifiers())) {
				throw new ClassReflectionException(clz, "Class should only have a public constructors!");
			}
		}
	}
	
	private static Set<Class<?>> simpleTypes = makeSet(Boolean.class, Integer.class, String.class, Pulse.class);
	
	/**
	 * Simple types are either Booleans, Integers, Strings, or Pulses.
	 */
	public static boolean isSimpleTypeClz(Class<?> clz) {
		for (Class<?> simpleType : simpleTypes) {
			if (simpleType == clz) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Simple types are either Booleans, Integers, Strings, or Pulses.
	 */
	public static void confirmSimpleTypeClz(Class<?> clz) throws ReflectionException {
		if (!isSimpleTypeClz(clz)) {
			String msg = "Invalid class type \"" + clz.getCanonicalName() + "\"; should be one of the following:";
			
			for (Class<?> simpleType : simpleTypes) {
				msg += "\n\t" + simpleType.getCanonicalName();
			}
			
			throw new ClassReflectionException(clz, msg);
		}
	}
	
	public static void confirmNoNestedClasses(Class<?> clz) throws ReflectionException {
		if (clz.getDeclaredClasses().length > 0) {
			throw new ClassReflectionException(clz, "Class should NOT declare nested classes!");
		}
	}
	
	public static void confirmSuperclass(Class<?> clz, Object... additionalAllowedSuperclzList) throws ReflectionException {
		confirmSuperclass(clz, new HashSet<Class<?>>(), additionalAllowedSuperclzList);
	}
	
	public static void confirmSuperclass(Class<?> clz, Set<Class<?>> allowedSuperclasses, Object... additionalAllowedSuperclzList) throws ReflectionException {
		if (clz.getSuperclass() == null) {
			throw new ClassReflectionException(clz, "Class does not have a super-class!");
		}
		
		if (!allowedSuperclasses.contains(clz.getSuperclass())) {
			if (!makeSet(additionalAllowedSuperclzList).contains(clz.getSuperclass())) {
				String msg = "Invalid immediate super-class \"" + clz.getSuperclass().getCanonicalName() + "\"; should be one of the following:";
				
				for (Class<?> c : allowedSuperclasses) {
					msg += "\n\t" + c.getCanonicalName();
				}
				
				for (Object c : additionalAllowedSuperclzList) {
					msg += "\n\t" + ((Class<?>)c).getCanonicalName();
				}
				
				throw new ClassReflectionException(clz, msg);
			}
		}
	}
	
	public static void confirmInheritsFrom(Class<?> clz, Object... allowedSuperclzList) throws ReflectionException {
		for (Object superClz : allowedSuperclzList) {
			Class<?> c = (Class<?>)superClz;
			
			if (c.isAssignableFrom(clz)) {
				return;
			}
		}
		
		String msg = "Missing super-class; should inherit from one of the following:";
		
		for (Object superClz : allowedSuperclzList) {
			Class<?> c = (Class<?>)superClz;
			msg += "\n\t" + c.getCanonicalName();
		}
		
		throw new ClassReflectionException(clz, msg);
	}
	
	public static void confirmStdFields(Class<?> clz, ModReq.StaticReq staticReq, Object... allowedTypeList) throws ReflectionException {
		confirmStdFields(clz, staticReq, null, allowedTypeList);
	}
	
	public static void confirmStdFields(Class<?> clz, ModReq.StaticReq staticReq, Predicate<Class<?>> pred, Object... allowedTypeList) throws ReflectionException {
		for (Field f : getAllFields(clz)) {
			FieldUtils.confirmStdField(f, staticReq);
			
			if (pred != null && pred.test(f.getType())) {
				continue;
			}
			
			if (FieldUtils.hasAllowedType(f, allowedTypeList)) {
				continue;
			}
			
			if (allowedTypeList.length > 0) {
				String msg = "Illegal field type \"" + f.getType().getCanonicalName() + "\"; should be one of the following:";
				
				for (Object allowedType : allowedTypeList) {
					msg += "\n\t" + ((Class<?>)allowedType).getCanonicalName();
				}
				
				throw new FieldReflectionException(f, msg);
			} else {
				if (pred != null) {
					throw new FieldReflectionException(f, "Field fails custom predicate!");
				}
				
				throw new ClassReflectionException(clz, "Class should not have any fields!");
			}
		}
	}
	
	public static void confirmAssignedFields(Class<?> clz, ModReq.AssignedReq assignedReq) throws ReflectionException {
		for (Field f : getDeclaredFields(clz)) {
			FieldUtils.confirmAssigned(f, assignedReq);
		}
	}
	
	public static void confirmInterfaces(Class<?> clz, Object... allowedInterfaceList) throws ReflectionException {
		if (allowedInterfaceList.length > 0) {
			Set<Class<?>> allowedInterfaces = makeSet(allowedInterfaceList);
			
			for (Class<?> i : clz.getInterfaces()) {
				if (!allowedInterfaces.contains(i)) {
					String msg = "Illegal interface \"" + i.getCanonicalName() + "\n; should be one of the following:";
					
					for (Object allowedInterface : allowedInterfaces) {
						msg += "\n\t" + ((Class<?>)allowedInterface).getCanonicalName();
					}
					
					throw new ClassReflectionException(clz, msg);
				}
			}
		} else {
			if (clz.getInterfaces().length > 0) {
				throw new ClassReflectionException(clz, "No interfaces allowed!");
			}
		}
	}
	
	@SuppressWarnings("unchecked")
	public static <T> T createStdInstance(Class<T> clz) throws ReflectionException {
		int m = clz.getModifiers();
		
		try {
			if (!Modifier.isStatic(m) && clz.getDeclaringClass() != null) {
				Constructor<?> c = clz.getConstructor(clz.getDeclaringClass());
				return (T)c.newInstance(createStdInstance(clz.getDeclaringClass()));
			} else {
				return clz.getConstructor().newInstance();
			}
		} catch (NoSuchMethodException | InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
			throw new ClassReflectionException(clz, e);
		}
	}
}
