package lib.sim;

import java.util.*;

import lib.asal.*;
import lib.asal.parsing.ASALException;
import lib.asal.parsing.api.*;
import lib.behave.*;
import lib.behave.proto.*;
import lib.blocks.models.UnifyingBlock;
import lib.blocks.models.UnifyingBlock.*;
import lib.utils.*;

public class JavaSimPrinter extends AbstractJavaSimPrinter {
	public JavaSimPrinter(UnifyingBlock target) throws ASALException {
		super(target);
	}
	
	private static String getTypeText(ASALDataType t) {
		switch (t) {
			case BOOLEAN:
				return "Boolean";
			case NUMBER:
				return "Integer";
			case PULSE:
				return "Boolean";
			case STRING:
				return "String";
			case VOID:
				return "void";
			default:
				throw new Error("Should not happen!");
		}
	}
	
	private static String getValueText(TritoStateMachine sm, ASALVariable<?> v) {
		if (sm != null) {
			ASALLiteral lit = sm.initialValuation.get(v.getName());
			
			if (lit != null) {
				switch (v.getType()) {
					case BOOLEAN:
					case PULSE:
						return Boolean.toString(lit.getText().equals("TRUE"));
					case NUMBER:
					case STRING:
						return lit.getText();
					case VOID:
						throw new Error("Should not happen!");
					default:
						throw new Error("Should not happen!");
				}
			}
		}
		
		switch (v.getType()) {
			case BOOLEAN:
				return "false";
			case NUMBER:
				return "0";
			case PULSE:
				return "false";
			case STRING:
				System.err.println("Variable " + v.getName() + " has not been initialized!");
				return "\"\"";
				//throw new Error("Should not happen!");
			case VOID:
				throw new Error("Should not happen!");
			default:
				throw new Error("Should not happen!");
		}
	}
	
	private static <T> String concat(String prefix, String sep, String suffix, Collection<? extends T> items, java.util.function.Function<T, String> fct) {
		Iterator<? extends T> q = items.iterator();
		
		if (q.hasNext()) {
			String result = fct.apply(q.next());
			
			while (q.hasNext()) {
				result += sep + fct.apply(q.next());
			}
			
			return prefix + result + suffix;
		}
		
		return prefix + suffix;
	}
	
	private boolean isExposedVar(ASALVariable<?> v) {
		for (ReprFlow rf : target.reprFlows) {
			if (rf.source.stateMachineVar == v || rf.target.stateMachineVar == v) {
				return false;
			}
		}
		return true;
	}
	
	private static boolean doesVertexContainVertex(TritoVertex root, TritoVertex requiredVertex) {
		if (root == requiredVertex) {
			return true;
		}
		
		for (TritoVertex v : root.getChildVertices()) {
			if (doesVertexContainVertex(v, requiredVertex)) {
				return true;
			}
		}
		
		return false;
	}
	
	private static boolean containsAllChildren(Collection<? extends TritoVertex> vertices, TritoVertex parent) {
		for (TritoVertex child : parent.getChildVertices()) {
			if (!vertices.contains(child)) {
				return false;
			}
		}
		
		return true;
	}
	
	private static List<TritoVertex> sortVerticesByRank(Collection<? extends TritoVertex> vertices) {
		List<TritoVertex> remaining = new ArrayList<TritoVertex>();
		remaining.addAll(vertices);
		
		List<TritoVertex> result = new ArrayList<TritoVertex>();
		
		while (remaining.size() > 0) {
			int oldSize = remaining.size();
			
			for (int index = 0; index < remaining.size(); ) {
				if (containsAllChildren(result, remaining.get(index))) {
					result.add(remaining.remove(index));
				} else {
					index++;
				}
			}
			
			if (remaining.size() == oldSize) {
				throw new Error("Should not happen; no progress!");
			}
		}
		
		return result;
	}
	
	private static boolean isFinalVertex(TritoVertex v) {
		return FinalVertex.class.isAssignableFrom(v.getClz());
	}
	
	private static boolean isStateMachineVertex(TritoVertex v) {
		return StateMachine.class.isAssignableFrom(v.getClz());
	}
	
	private static boolean isStateVertex(TritoVertex v) {
		return State.class.isAssignableFrom(v.getClz()) && !isStateMachineVertex(v);
	}
	
	private List<List<TritoTransition>> getRouteUntilStableOrFinal(TritoStateMachine sm, Collection<? extends TritoVertex> startingVertices) {
		List<List<TritoTransition>> result = new ArrayList<List<TritoTransition>>();
		
		List<List<TritoTransition>> fringe = new ArrayList<List<TritoTransition>>();
		List<List<TritoTransition>> newFringe = new ArrayList<List<TritoTransition>>();
		
		for (TritoTransition t : sm.transitions) {
			if (startingVertices.contains(t.getSourceVertex())) {
				List<TritoTransition> ts = new ArrayList<TritoTransition>();
				ts.add(t);
				
				if (t.isLocal() || isStateVertex(t.getTargetVertex()) || isFinalVertex(t.getTargetVertex())) {
					result.add(ts);
				} else {
					fringe.add(ts);
				}
			}
		}
		
		while (fringe.size() > 0) {
			newFringe.clear();
			
			for (List<TritoTransition> f : fringe) {
				TritoVertex tgt = f.get(f.size() - 1).getTargetVertex();
				
				for (TritoTransition t : sm.transitions) {
					if (t.getSourceVertex() == tgt) {
						if (f.contains(t)) {
							String s = "Infinite loop in transitions of " + sm.clz.getCanonicalName() + ":";
							
							for (int index = 0; index < f.size(); index++) {
								s += "\n\tTransition " + index + ":\n\t\t\t" + transitionToDebugText(f.get(index)).replace("\n", "\n\t\t\t");
							}
							
							s += "\n\tLooping transition " + f.size() + ":\n\t\t\t" + transitionToDebugText(t).replace("\n", "\n\t\t\t");
							throw new Error(s);
						}
						
						List<TritoTransition> ts = new ArrayList<TritoTransition>();
						ts.addAll(f);
						ts.add(t);
						
						if (isStateVertex(t.getTargetVertex()) || isFinalVertex(t.getTargetVertex())) {
							result.add(ts);
						} else {
							newFringe.add(ts);
						}
					}
				}
			}
			
			fringe.clear();
			fringe.addAll(newFringe);
		}
		
		return result;
	}
	
	private String transitionToDebugText(TritoTransition t) {
		String result = "Source: " + getName(t.getSourceVertex());
		result += "\nTarget: " + getName(t.getTargetVertex());
		
		if (t.getEvent() != null) {
			result += "\nEvent: " + t.getEvent().textify(LOD.FULL);
		}
		
		if (t.getGuard() != null) {
			result += "\nGuard: " + t.getGuard().textify(LOD.FULL);
		}
		
		if (t.getStatement() != null) {
			result += "\nStatement: " + t.getStatement().textify(LOD.FULL).replace("\n", " ");
		}
		
		return result;
	}
	
	private void printReprBlock(ReprBlock rb) {
		TritoStateMachine sm = rb.getOwnedStateMachine() != null ? rb.getOwnedStateMachine().representedStateMachine : null;
		Set<TritoVertex> stateVertices = new HashSet<TritoVertex>();
		
		if (sm != null) {
			stateVertices.addAll(sm.vertices);
			stateVertices.removeIf(x -> { return !isStateVertex(x); });
		}
		
		println("public static class " + getName(rb.type) + " {", +1);
		
		// Field declarations:
		println("private final Set<String> vertices;");
		
		if (sm != null) {
			for (Map.Entry<String, ASALVariable<?>> entry : sm.inPortVars.entrySet()) {
				if (isExposedVar(entry.getValue())) {
					println("public final In<" + getTypeText(entry.getValue().getType()) + "> " + getName(entry.getValue()) + ";");
				} else {
					println("public final Var<" + getTypeText(entry.getValue().getType()) + "> " + getName(entry.getValue()) + ";");
				}
			}
			
			for (Map.Entry<String, ASALVariable<?>> entry : sm.outPortVars.entrySet()) {
				if (isExposedVar(entry.getValue())) {
					println("public final Out<" + getTypeText(entry.getValue().getType()) + "> " + getName(entry.getValue()) + ";");
				} else {
					println("public final Var<" + getTypeText(entry.getValue().getType()) + "> " + getName(entry.getValue()) + ";");
				}
			}
			
			for (Map.Entry<String, ASALVariable<?>> entry : sm.stateMachineVars.entrySet()) {
				println("public final Var<" + getTypeText(entry.getValue().getType()) + "> " + getName(entry.getValue()) + ";");
			}
		} else {
			for (ReprPort rp : rb.ownedPorts) {
				if (rp.getDir() == Dir.OUT) {
					println("public final Out<" + getTypeText(rp.getType()) + "> " + getName(rp) + ";");
				} else {
					println("public final In<" + getTypeText(rp.getType()) + "> " + getName(rp) + ";");
				}
			}
		}
		
		// Constructor that initialize the component state:
		println("public " + getName(rb.type) + "() {", +1);
		println("vertices = new HashSet<String>();");
		
		if (sm != null) {
			for (Map.Entry<String, ASALVariable<?>> entry : sm.inPortVars.entrySet()) {
				if (isExposedVar(entry.getValue())) {
					println(getName(entry.getValue()) + " = new In<" + getTypeText(entry.getValue().getType()) + ">(" + getValueText(sm, entry.getValue()) + ");");
				} else {
					println(getName(entry.getValue()) + " = new Var<" + getTypeText(entry.getValue().getType()) + ">(" + getValueText(sm, entry.getValue()) + ");");
				}
			}
			
			for (Map.Entry<String, ASALVariable<?>> entry : sm.outPortVars.entrySet()) {
				if (isExposedVar(entry.getValue())) {
					println(getName(entry.getValue()) + " = new Out<" + getTypeText(entry.getValue().getType()) + ">(" + getValueText(sm, entry.getValue()) + ");");
				} else {
					println(getName(entry.getValue()) + " = new Var<" + getTypeText(entry.getValue().getType()) + ">(" + getValueText(sm, entry.getValue()) + ");");
				}
			}
			
			for (Map.Entry<String, ASALVariable<?>> entry : sm.stateMachineVars.entrySet()) {
				println(getName(entry.getValue()) + " = new Var<" + getTypeText(entry.getValue().getType()) + ">(" + getValueText(sm, entry.getValue()) + ");");
			}
		} else {
			for (ReprPort rp : rb.ownedPorts) {
				if (rp.getDir() == Dir.OUT) {
					println(getName(rp) + " = new Out<" + getTypeText(rp.getType()) + ">(" + getValueText(null, rp) + ");");
				} else {
					println(getName(rp) + " = new In<" + getTypeText(rp.getType()) + ">(" + getValueText(null, rp) + ");");
				}
			}
		}
		
		println(-1, "}");
		
		// Function for copying the block state:
		println("public " + getName(rb.type) + " copyFrom(" + getName(rb.type) + " source) {", +1);
		println("vertices.clear();");
		println("vertices.addAll(source.vertices);");
		
		if (sm != null) {
			for (Map.Entry<String, ASALVariable<?>> entry : sm.inPortVars.entrySet()) {
				println(getName(entry.getValue()) + ".value = source." + getName(entry.getValue()) + ".value;");
			}
			
			for (Map.Entry<String, ASALVariable<?>> entry : sm.outPortVars.entrySet()) {
				println(getName(entry.getValue()) + ".value = source." + getName(entry.getValue()) + ".value;");
			}
			
			for (Map.Entry<String, ASALVariable<?>> entry : sm.stateMachineVars.entrySet()) {
				println(getName(entry.getValue()) + ".value = source." + getName(entry.getValue()) + ".value;");
			}
		} else {
			for (ReprPort rp : rb.ownedPorts) {
				println(getName(rp) + ".value = source." + getName(rp) + ".value;");
			}
		}
		
		println("return this;");
		println(-1, "}");
		
		if (sm != null) {
			println("public static Set<" + getName(rb.type) + "> create() {");
			println("\treturn new HashSet<" + getName(rb.type) + ">(new " + getName(rb.type) + "().enterState(\"" + getName(sm.rootVertex) + "\"));");
			println("}");
		} else {
			println("public static Set<" + getName(rb.type) + "> create() {");
			println("\treturn new HashSet<" + getName(rb.type) + ">();");
			println("}");
		}
		
		// Retrieve state configuration:
		println("public Set<String> getVertices() {");
		println("\treturn Collections.unmodifiableSet(vertices);");
		println("}");
		
		// ASAL functions become Java functions:
		if (sm != null) {
			for (ASALFunction f : sm.functions.values()) {
				String header = "private " + getTypeText(f.getReturnType()) + " " + getName(f) + "(";
				
				if (f.getParams().size() > 0) {
					header += getTypeText(f.getParams().get(0).getType()) + " " + getName(f.getParams().get(0));
					
					for (int index = 1; index < f.getParams().size(); index++) {
						header += ", " + getTypeText(f.getParams().get(index).getType()) + " " + getName(f.getParams().get(index));
					}
				}
				
				header += ") {";
				println(header);
				
				ASAL2JavaSimVisitor v = new ASAL2JavaSimVisitor(this, "this", f.createContext(sm));
				
				try {
					for (String s : v.visitStat(f.getBody())) {
						println("\t" + s);
					}
				} catch (ASALException e) {
					throw new Error(e);
				}
				
				println("}");
			}
		}
		
		// Function that determine if a transition is enabled:
		if (sm != null) {
			println("private boolean isTransitionEnabled(String id, boolean checkVertices) {");
			
			if (sm.transitions.size() > 0) {
				println("\tswitch (id) {");
				
				for (TritoTransition t : sm.transitions) {
					println("\t\t/* Source: " + getName(t.getSourceVertex()) + " */");
					println("\t\t/* Target: " + getName(t.getTargetVertex()) + " */");
					
					if (t.getEvent() != null) {
						println("\t\t/* Event: " + t.getEvent().textify(LOD.FULL) + " */");
					}
					
					if (t.getGuard() != null) {
						println("\t\t/* Guard: " + t.getGuard().textify(LOD.FULL) + " */");
					}
					
					if (t.getStatement() != null) {
						println("\t\t/* Statement: " + t.getStatement().textify(LOD.FULL).replace("\n", " ") + " */");
					}
					
					ASAL2JavaSimVisitor v = new ASAL2JavaSimVisitor(this, "this", sm);
					println("\t\tcase \"" + getName(t) + "\":");
					
					try {
						println("\t\t\tif (checkVertices && !vertices.contains(\"" + getName(t.getSourceVertex()) + "\")) return false;");
						
						if (t.getEvent() != null) {
							if (t.getEvent() instanceof ASALTrigger) {
								ASALTrigger trigger = (ASALTrigger)t.getEvent();
								println("\t\t\tif (" + v.visitExpr(trigger.getExpr()).get(0) + ") return false;");
							}
							
							if (t.getEvent() instanceof ASALCall) {
								println("\t\t\t// ... call event ...");
							}
							
							if (t.getEvent() instanceof ASALTimeout) {
								println("\t\t\t// ... timeout event ...");
							}
						}
						
						if (t.getGuard() != null) {
							println("\t\t\tif (" + v.visitExpr(t.getGuard()).get(0) + ") return false;");
						}
					} catch (ASALException e) {
						throw new Error(e);
					}
					
					println("\t\t\treturn true;");
				}
				
				println("\t\tdefault:");
				println("\t\t\tthrow new Error(\"Should not happen; transition \" + id + \" not found!\");");
				println("\t}");
			}
			
			println("}");
		}
		
		// Function for entering states.
		// Returns all possible stable states that are eventually reached:
		if (sm != null) {
			println("private Set<" + getName(rb.type) + "> enterState(String id) {");
			println("\tSet<" + getName(rb.type) + "> result = new HashSet<" + getName(rb.type) + ">();");
			println("\t" + getName(rb.type) + " e0 = new " + getName(rb.type) + "().copyFrom(this);");
			
			if (stateVertices.size() > 0) {
				println("\tswitch (id) {");
				
				for (TritoVertex v : sm.vertices) {
					if (stateVertices.contains(v) || isStateMachineVertex(v)) {
						println("\t\tcase \"" + getName(v) + "\":");
						println("\t\t\tif (vertices.add(\"" + getName(v) + "\")) {", +4);
						
						if (v.getOnEntry() != null) {
							ASAL2JavaSimVisitor vis = new ASAL2JavaSimVisitor(this, "e0", sm);
							
							try {
								for (String s : vis.visitStat(v.getOnEntry().getStatement())) {
									println(s);
								}
							} catch (ASALException e) {
								throw new Error(e);
							}
						}
						
						if (v.getInitialVertices().size() > 0) {
							println("List<List<List<String>>> routesPerInitialVertex = " + concat("Arrays.asList(", ", ", ")", v.getInitialVertices(), (iv) -> {
								List<List<TritoTransition>> tss = getRouteUntilStableOrFinal(sm, Collections.singleton(iv));
								return concat("Arrays.asList(", ", ", ")", tss, (ts) -> {
									return concat("Arrays.asList(", ", ", ")", ts, (t) -> {
										return "\"" + getName(t) + "\"";
									});
								});
							}) + ";");
							
							println("for (List<List<List<String>>> rts : getAllOrderings(routesPerInitialVertex)) {");
							println("\tfor (List<List<String>> rt : getAllPermutations(rts)) {");
							println("\t\tboolean add = true;");
							println("\t\tfor (List<String> r : rt) {");
							println("\t\t\tfor (String step : r) {");
							println("\t\t\t\tif (!e0.isTransitionEnabled(step, false)) {");
							println("\t\t\t\t\tadd = false;");
							println("\t\t\t\t\tbreak;");
							println("\t\t\t\t}");
							println("\t\t\t}");
							println("\t\t\tif (!add) break;");
							println("\t\t}");
							println("\t\tif (add) {");
							println("\t\t\tSet<" + getName(rb.type) + "> fringe = new HashSet<" + getName(rb.type) + ">();");
							println("\t\t\tSet<" + getName(rb.type) + "> newFringe = new HashSet<" + getName(rb.type) + ">();");
							println("\t\t\tfringe.add(new " + getName(rb.type) + "().copyFrom(e0));");
							println("\t\t\tfor (List<String> r : rt) {");
							println("\t\t\t\tfor (String step : r) {");
							println("\t\t\t\t\tnewFringe.clear();");
							println("\t\t\t\t\tfor (" + getName(rb.type) + " f : fringe) {");
							println("\t\t\t\t\t\tnewFringe.addAll(f.fireTransition(step));");
							println("\t\t\t\t\t}");
							println("\t\t\t\t\tfringe.clear();");
							println("\t\t\t\t\tfringe.addAll(newFringe);");
							println("\t\t\t\t}");
							println("\t\t\t}");
							println("\t\t\tresult.addAll(fringe);");
							println("\t\t}");
							println("\t}");
							println("}");
						} else {
							println("result.add(e0);");
						}
						
						println(-4, "\t\t\t}");
						println("\t\t\tbreak;");
					}
				}
				
				println("\t\tdefault:");
				println("\t\t\tthrow new Error(\"Should not happen; state \" + id + \" not found!\");");
				println("\t}");
			}
			
			println("\tif (result.isEmpty()) result.add(e0);");
			println("\treturn result;");
			println("}");
		}
		
		// Function for exiting states.
		// Returns all possible stable states that are eventually reached:
		if (sm != null) {
			println("private Set<" + getName(rb.type) + "> exitState(String id) {");
			println("\tSet<" + getName(rb.type) + "> result = new HashSet<" + getName(rb.type) + ">();");
			println("\t" + getName(rb.type) + " e0 = new " + getName(rb.type) + "();");
			
			if (stateVertices.size() > 0) {
				println("\tswitch (id) {");
				
				for (TritoVertex v : stateVertices) {
					println("\t\tcase \"" + getName(v) + "\":");
					println("\t\t\tif (vertices.remove(\"" + getName(v) + "\")) {");
					
					List<TritoVertex> childStates = new ArrayList<TritoVertex>();
					
					for (TritoVertex cv : v.getChildVertices()) {
						if (isStateVertex(cv)) {
							childStates.add(cv);
						}
					}
					
					if (childStates.size() > 0) {
						println("\t\t\t\tList<String> states = new ArrayList<String>();");
						
						for (TritoVertex w : childStates) {
							println("\t\t\t\tstates.add(\"" + getName(w) + "\");");
						}
						
						for (int index = 0; index < childStates.size(); index++) {
							println("\t\t\t\tfor (" + getName(rb.type) + " e" + (index + 1) + " : e" + index + ".exitState(states.get(" + index + "))) {", +1);
						}
						
						if (v.getOnExit() != null) {
							ASAL2JavaSimVisitor vis = new ASAL2JavaSimVisitor(this, "e" + childStates.size(), sm);
							
							try {
								for (String s : vis.visitStat(v.getOnExit().getStatement())) {
									println("\t\t\t\t" + s);
								}
							} catch (ASALException e) {
								throw new Error(e);
							}
						}
						
						println("\t\t\t\tresult.add(e" + childStates.size() + ");");
						
						for (int index = childStates.size() - 1; index >= 0; index--) {
							println(-1, "\t\t\t\t}");
						}
					} else {
						if (v.getOnExit() != null) {
							ASAL2JavaSimVisitor vis = new ASAL2JavaSimVisitor(this, "e0", sm);
							
							try {
								for (String s : vis.visitStat(v.getOnExit().getStatement())) {
									println("\t\t\t\t" + s);
								}
							} catch (ASALException e) {
								throw new Error(e);
							}
						}
						
						println("\t\t\t\tresult.add(e0);");
					}
					
					println("\t\t\t}");
					println("\t\t\tbreak;");
				}
				
				println("\t\tdefault:");
				println("\t\t\tthrow new Error(\"Should not happen; state \" + id + \" not found!\");");
				println("\t}"); // (End of switch.)
			}
			
			println("\tif (result.isEmpty()) result.add(e0);");
			println("\treturn result;");
			println("}");
		}
		
		// We fire transitions to final vertices immediately.
		// Here, we check if all regions of a composite state have reached a final vertex.
		// If so, we should fire an outgoing transition.
		if (sm != null) {
			println("private Set<" + getName(rb.type) + "> firePostExitTransitions() {");
			println("\tSet<" + getName(rb.type) + "> result = new HashSet<" + getName(rb.type) + ">();");
			
			for (TritoVertex v : sortVerticesByRank(sm.vertices)) {
				if (v.getChildVertices().size() > 0) {
					List<List<TritoTransition>> tss = getRouteUntilStableOrFinal(sm, Collections.singleton(v));
					
					//Exit routes may not start with a guard/event/...:
					tss.removeIf(ts -> {
						if (ts.size() > 0) {
							if (ts.get(0).getGuard() != null || ts.get(0).getEvent() != null) {
								return true;
							}
						}
						return false;
					});
					
					if (tss.size() > 0) {
						String condition = "vertices.contains(\"" + getName(v) + "\")";
						
						for (TritoVertex w : v.getChildVertices()) {
							if (!isFinalVertex(w)) {
								condition += " && !vertices.contains(\"" + getName(w) + "\")";
							}
						}
						
						//Check if we should exit a composite state:
						println("\tif (" + condition + ") {");
						
						for (List<TritoTransition> ts : tss) {
							println("\t\tif (" + getCombinedCondition("", ts) + ") {");
							
							for (int index = 0; index < ts.size(); index++) {
								TritoTransition t = ts.get(index);
								println("\t\t\t\tfor (" + getName(rb.type) + " e" + (index + 1) + " : e" + index + ".fireTransition(\"" + getName(t) + "\")) {", +1);
							}
							
							println("\t\t\t\tresult.addAll(e" + ts.size() + ".firePostExitTransitions());");
							
							for (int index = ts.size() - 1; index >= 0; index--) {
								println(-1, "\t\t\t\t}");
							}
							
							println("\t\t}");
						}
						
						println("\t\tif (result.isEmpty()) throw new Error(\"There should be at least one enabled transition!\");");
						println("\t}");
						
						// All exit routes are blocked, so stay in the current state:
						println("\tresult.add(new " + getName(rb.type) + "());");
						println("\t\treturn result;");
					}
				}
			}
			
			// No composite states with exit routes found:
			println("\tresult.add(new " + getName(rb.type) + "());");
			println("\treturn result;");
			println("}"); // (End of firePostExitsTransitions.)
		}
		
		//Function that applies the effects of a transition:
		if (sm != null) {
			println("private Set<" + getName(rb.type) + "> fireTransition(String id) {");
			println("\tSet<" + getName(rb.type) + "> result = new HashSet<" + getName(rb.type) + ">();");
			
			if (sm.transitions.size() > 0) {
				println("\tswitch (id) {");
				
				for (TritoTransition t : sm.transitions) {
					println("\t\t/* Source: " + getName(t.getSourceVertex()) + " */");
					println("\t\t/* Target: " + getName(t.getTargetVertex()) + " */");
					
					if (t.getEvent() != null) {
						println("\t\t/* Event: " + t.getEvent().textify(LOD.FULL) + " */");
					}
					
					if (t.getGuard() != null) {
						println("\t\t/* Guard: " + t.getGuard().textify(LOD.FULL) + " */");
					}
					
					if (t.getStatement() != null) {
						println("\t\t/* Statement: " + t.getStatement().textify(LOD.FULL).replace("\n", " ") + " */");
					}
					
					println("\t\tcase \"" + getName(t) + "\":");
					
					if (t.isLocal()) {
						println("\t\t\t{");
						println("\t\t\t\t" + getName(rb.type) + " e0 = new " + getName(rb.type) + "().copyFrom(this);");
					} else {
						for (TritoVertex w : t.getTargetVertex().getParentVertex().getChildVertices()) {
							if (doesVertexContainVertex(w, t.getSourceVertex())) {
								println("\t\t\tfor (" + getName(rb.type) + " e0 : exitState(\"" + getName(w) + "\")) {");
								break; //There should be only one match, so break the loop!
							}
						}
					}
					
					ASAL2JavaSimVisitor v = new ASAL2JavaSimVisitor(this, "e0", sm);
					
					try {
						for (String s : v.visitStat(t.getStatement())) {
							println("\t\t\t\t" + s);
						}
					} catch (ASALException e) {
						throw new Error(e);
					}
					
					if (!t.isLocal() && isStateVertex(t.getTargetVertex())) {
						println("\t\t\t\tfor (" + getName(rb.type) + " e1 : e0.enterState(\"" + getName(t.getTargetVertex()) + "\")) {");
						println("\t\t\t\t\tresult.add(e1);");
						println("\t\t\t\t}");
					} else {
						println("\t\t\t\tresult.add(e0);");
					}
					
					println("\t\t\t}");
					println("\t\t\tbreak;");
				}
				
				println("\t\tdefault:");
				println("\t\t\tthrow new Error(\"Should not happen; transition \" + id + \" not found!\");");
				println("\t}");
			}
			
			println("\treturn result;");
			println("}"); // (End of fireTransition.)
		}
		
		//End of nested subclass:
		println(-1, "}");
	}
	
	private String getCombinedCondition(String prefix, List<TritoTransition> ts) {
		String combinedCondition = prefix + "isTransitionEnabled(\"" + getName(ts.get(0)) + "\", true)";
		
		for (int index = 1; index < ts.size(); index++) {
			combinedCondition += " && " + prefix + "isTransitionEnabled(\"" + getName(ts.get(index)) + "\", false)";
		}
		
		return combinedCondition;
	}
	
	@Override
	protected void print(String mode, Object object) {
		println("package lib.sim;");
		println("");
		println("import java.util.*;");
		println("");
		println("public class JavaSimClass {", +1);
		
		// Helper class:
		println("public static class Var<T> {");
		println("\tprivate T value;");
		println("\tpublic Var(T value) {");
		println("\t\tthis.value = value;");
		println("\t}");
		println("\tpublic Var(Var<T> source) {");
		println("\t\tvalue = source.value;");
		println("\t}");
		println("\tpublic final T get() {");
		println("\t\treturn value;");
		println("\t}");
		println("}");
		
		// Helper class:
		println("public static class In<T> {");
		println("\tprivate T value;");
		println("\tpublic In(T value) {");
		println("\t\tthis.value = value;");
		println("\t}");
		println("\tpublic In(In<T> source) {");
		println("\t\tvalue = source.value;");
		println("\t}");
		println("\tpublic final T get() {");
		println("\t\treturn value;");
		println("\t}");
		println("\tpublic final void set(T value) {");
		println("\t\tthis.value = value;");
		println("\t}");
		println("}");
		
		// Helper class:
		println("public static class Out<T> {");
		println("\tprivate T value;");
		println("\tpublic Out(T value) {");
		println("\t\tthis.value = value;");
		println("\t}");
		println("\tpublic Out(Out<T> source) {");
		println("\t\tvalue = source.value;");
		println("\t}");
		println("\tpublic final T get() {");
		println("\t\treturn value;");
		println("\t}");
		println("}");
		
		// Helper functions:
		println("private static <T> List<List<T>> getAllOrderings(List<T> elems) {");
		println("\tList<List<T>> result = new ArrayList<List<T>>();");
		println("\taddAllOrderings(elems, new ArrayList<T>(), result);");
		println("\treturn result;");
		println("}");
		println("private static <T> void addAllOrderings(List<T> elems, List<T> soFar, List<List<T>> dest) {");
		println("\tif (soFar.size() < elems.size()) {");
		println("\t\tfor (T elem : elems) {");
		println("\t\t\tif (!soFar.contains(elem)) {");
		println("\t\t\t\tList<T> newSoFar = new ArrayList<T>(soFar);");
		println("\t\t\t\tnewSoFar.add(elem);");
		println("\t\t\t\taddAllOrderings(elems, newSoFar, dest);");
		println("\t\t\t}");
		println("\t\t}");
		println("\t} else {");
		println("\t\tdest.add(soFar);");
		println("\t}");
		println("}");
		
		// Helper functions:
		println("private static <T> List<List<T>> getAllPermutations(List<List<T>> elems) {");
		println("\tList<List<T>> result = new ArrayList<List<T>>();");
		println("\taddAllPermutations(elems, new ArrayList<T>(), result);");
		println("\treturn result;");
		println("}");
		println("private static <T> void addAllPermutations(List<List<T>> elems, List<T> soFar, List<List<T>> dest) {");
		println("\tif (soFar.size() < elems.size()) {");
		println("\t\tfor (T elem : elems.get(soFar.size())) {");
		println("\t\t\tList<T> newSoFar = new ArrayList<T>(soFar);");
		println("\t\t\tnewSoFar.add(elem);");
		println("\t\t\taddAllPermutations(elems, newSoFar, dest);");
		println("\t\t}");
		println("\t} else {");
		println("\t\tdest.add(soFar);");
		println("\t}");
		println("}");
		
		// Classes that store the state of each block:
		for (ReprBlock rb : target.reprBlocks) {
			printReprBlock(rb);
		}
		
		// Fields for system components:
		for (ReprBlock rb : target.reprBlocks) {
			println("public final " + getName(rb.type) + " " + getName(rb) + ";");
		}
		
		// Constructor:
		println("public JavaSimClass() {");
		
		for (ReprBlock rb : target.reprBlocks) {
			println("\t" + getName(rb) + " = new " + getName(rb.type) + "();");
		}
		
		println("}");
		
		// Function for copying the system state:
		println("public JavaSimClass copyFrom(JavaSimClass source) {");
		
		for (ReprBlock rb : target.reprBlocks) {
			println("\t" + getName(rb) + ".copyFrom(source." + getName(rb) + ");");
		}
		
		println("\treturn this;");
		println("}");
		
		// Initial states:
		println("public Set<JavaSimClass> getInitialStates() {", +1);
		println("Set<JavaSimClass> result = new HashSet<JavaSimClass>();");
		
		for (int index = 0; index < target.reprBlocks.size(); index++) {
			ReprBlock rb = target.reprBlocks.get(index);
			println("for (" + getName(rb.type) + " e" + index + " : " + getName(rb.type) + ".create()) {", +1);
		}
		
		println("JavaSimClass a = new JavaSimClass();");
		
		for (int index = 0; index < target.reprBlocks.size(); index++) {
			ReprBlock rb = target.reprBlocks.get(index);
			println("a." + getName(rb) + ".copyFrom(e" + index + ");");
		}
		
		println("result.add(a);");
		
		for (int index = target.reprBlocks.size() - 1; index >= 0; index--) {
			println(-1, "}");
		}
		
		println("return result;");
		println(-1, "}");
		
		// Next states:
		println("public Set<JavaSimClass> getPossibleNextStates() {");
		println("\tSet<JavaSimClass> result = new HashSet<JavaSimClass>();");
		
		for (ReprBlock rb : target.reprBlocks) {
			TritoStateMachine sm = rb.getOwnedStateMachine() != null ? rb.getOwnedStateMachine().representedStateMachine : null;
			
			if (sm != null) {
				Set<TritoVertex> startingVertices = new HashSet<TritoVertex>();
				
				for (TritoTransition t : sm.transitions) {
					if (isStateVertex(t.getSourceVertex())) {
						if (t.getSourceVertex().getChildVertices().isEmpty() || t.getGuard() != null || t.getEvent() != null) {
							startingVertices.add(t.getSourceVertex());
						}
					}
				}
				
				for (List<TritoTransition> ts : getRouteUntilStableOrFinal(sm, startingVertices)) {
					println("\tif (" + getCombinedCondition(getName(rb) + ".", ts) + ") {");
					println("\t\t" + getName(rb.type) + " e0 = new " + getName(rb.type) + "().copyFrom(" + getName(rb) + ");");
					
					for (int index = 0; index < ts.size(); index++) {
						TritoTransition t = ts.get(index);
						println("\t\tfor (" + getName(rb.type) + " e" + (index + 1) + " : e" + index + ".fireTransition(\"" + getName(t) + "\")) {", +1);
					}
					
					println("\t\tfor (" + getName(rb.type) + " e" + (ts.size() + 1) + " : e" + ts.size() + ".firePostExitTransitions()) {", +1);
					println("\t\tJavaSimClass a = new JavaSimClass().copyFrom(this);");
					println("\t\ta." + getName(rb) + ".copyFrom(e" + (ts.size() + 1) + ");");
					println("\t\tresult.add(a);");
					println(-1, "\t\t}");
					
					for (int index = ts.size() - 1; index >= 0; index--) {
						println(-1, "\t\t}");
					}
					
					println("\t}");
				}
			}
		}
		
		for (ReprFlow rf : target.reprFlows) {
			String src = getName(rf.source.owner) + "." + getName(rf.source) + ".value";
			String tgt = getName(rf.target.owner) + "." + getName(rf.target) + ".value";
			
			println("\tif (!" + tgt + ".equals(" + src + ")) {");
			println("\t\tJavaSimClass a = new JavaSimClass().copyFrom(this);");
			println("\t\ta." + tgt + " = a." + src + ";");
			println("\t\tresult.add(a);");
			println("\t}");
		}
		
		println("\treturn result;");
		println("}");
		println(-1, "}");
	}
}





