package lib.sim;

import java.util.*;

import lib.asal.*;
import lib.asal.parsing.ASALException;
import lib.behave.proto.*;
import lib.blocks.models.UnifyingBlock;
import lib.blocks.printing.AbstractPrinter;
import lib.utils.UnusedNames;

public abstract class AbstractJavaSimPrinter extends AbstractPrinter<UnifyingBlock> {
	protected final Set<String> varNames;
	protected final Set<String> funcNames;
	protected final Set<String> stateNames;
	protected final Set<String> beqNames;
	
	protected final Map<String, String> customKeywords;
	
	public AbstractJavaSimPrinter(UnifyingBlock target) throws ASALException {
		super(target);
		
		registerKeywords();
		customKeywords = getCustomKeywords();
		
		varNames = extractVarNames();
		funcNames = extractFuncNames();
		stateNames = extractStateNames();
		extractTransitionNames();
		beqNames = extractElemNames();
	}
	
	private void registerKeywords() {
		getUnusedName("abstract");
		getUnusedName("continue");
		getUnusedName("for");
		getUnusedName("new");
		getUnusedName("switch");
		getUnusedName("assert");
		getUnusedName("default");
		getUnusedName("package");
		getUnusedName("synchronized");
		getUnusedName("boolean");
		getUnusedName("do");
		getUnusedName("if");
		getUnusedName("private");
		getUnusedName("this");
		getUnusedName("break");
		getUnusedName("double");
		getUnusedName("implements");
		getUnusedName("protected");
		getUnusedName("throw");
		getUnusedName("byte");
		getUnusedName("else");
		getUnusedName("import");
		getUnusedName("public");
		getUnusedName("throws");
		getUnusedName("case");
		getUnusedName("enum");
		getUnusedName("instanceof");
		getUnusedName("return");
		getUnusedName("transient");
		getUnusedName("catch");
		getUnusedName("extends");
		getUnusedName("int");
		getUnusedName("short");
		getUnusedName("try");
		getUnusedName("char");
		getUnusedName("final");
		getUnusedName("interface");
		getUnusedName("static");
		getUnusedName("void");
		getUnusedName("class");
		getUnusedName("finally");
		getUnusedName("long");
		getUnusedName("strictfp");
		getUnusedName("volatile");
		getUnusedName("const");
		getUnusedName("float");
		getUnusedName("native");
		getUnusedName("super");
		getUnusedName("while");
	}
	
	@Override
	protected void println(String s) {
		String newS = s;
		
		for (Map.Entry<String, String> entry : customKeywords.entrySet()) {
			newS = newS.replace(entry.getKey(), entry.getValue());
		}
		
		super.println(newS);
	}
	
	private Map<String, String> getCustomKeywords() {
		Map<String, String> result = new HashMap<String, String>();
		
		//We reserve the following identifiers for our own goals:
		addCustomKeyword(result, "vertices");
		addCustomKeyword(result, "getVertices");
		addCustomKeyword(result, "Collections");
		addCustomKeyword(result, "List");
		addCustomKeyword(result, "ArrayList");
		addCustomKeyword(result, "Set");
		addCustomKeyword(result, "HashSet");
		addCustomKeyword(result, "JavaSimClass"); // Susceptible to change!!
		addCustomKeyword(result, "Object");
		addCustomKeyword(result, "Boolean");
		addCustomKeyword(result, "String");
		addCustomKeyword(result, "boolean");
		addCustomKeyword(result, "int");
		addCustomKeyword(result, "Var");
		addCustomKeyword(result, "In");
		addCustomKeyword(result, "Out");
		addCustomKeyword(result, "source");
		addCustomKeyword(result, "checkVertices");
		addCustomKeyword(result, "id");
		addCustomKeyword(result, "fireTransition");
		addCustomKeyword(result, "isTransitionEnabled");
		addCustomKeyword(result, "enterState");
		addCustomKeyword(result, "exitState");
		addCustomKeyword(result, "firePostExitTransitions");
		addCustomKeyword(result, "getPossibleNextStates");
		addCustomKeyword(result, "Math");
		addCustomKeyword(result, "routeList");
		addCustomKeyword(result, "route");
		addCustomKeyword(result, "rt");
		addCustomKeyword(result, "result");
		
		return result;
	}
	
	private void addCustomKeyword(Map<String, String> dest, String value) {
		addCustomKeyword(dest, value, value);
	}
	
	private void addCustomKeyword(Map<String, String> dest, String key, String value) {
		if (dest.containsKey(key)) {
			throw new Error("Duplicate custom keyword entry: " + key);
		}
		
		dest.put("{{{" + key + "}}}", getUnusedName(value));
	}
	
	private static void addVarName(Map<String, Set<ASALVariable<?>>> varsPerName, String varName, ASALVariable<?> var) {
		Set<ASALVariable<?>> vars = varsPerName.get(varName);
		
		if (vars == null) {
			vars = new HashSet<ASALVariable<?>>();
			varsPerName.put(varName, vars);
		}
		
		vars.add(var);
	}
	
	private Set<String> extractVarNames() {
		Map<String, Set<ASALVariable<?>>> varsPerName = new HashMap<String, Set<ASALVariable<?>>>();
		
		for (UnifyingBlock.ReprBlock rb : target.reprBlocks) {
			for (UnifyingBlock.ReprPort rp : rb.ownedPorts) {
				addVarName(varsPerName, rp.getName(), rp);
			}
			
			if (rb.getOwnedStateMachine() != null) {
				UnifyingBlock.ReprStateMachine rsm =  rb.getOwnedStateMachine();
				
				for (Map.Entry<String, ASALVariable<?>> entry : rsm.representedStateMachine.stateMachineVars.entrySet()) {
					addVarName(varsPerName, entry.getKey(), entry.getValue());
				}
				
				//Ports should have already been added, but
				//they have a separate state machine declaration that we want to reference with the same id:
				for (Map.Entry<String, ASALVariable<?>> entry : rsm.representedStateMachine.inPortVars.entrySet()) {
					addVarName(varsPerName, entry.getKey(), entry.getValue());
				}
				
				for (Map.Entry<String, ASALVariable<?>> entry : rsm.representedStateMachine.outPortVars.entrySet()) {
					addVarName(varsPerName, entry.getKey(), entry.getValue());
				}
			}
		}
		
		for (UnifyingBlock.ReprStateMachine rsm : target.reprStateMachines) {
			for (ASALFunction f : rsm.representedStateMachine.functions.values()) {
				for (ASALVariable<?> v : f.getParams()) {
					addVarName(varsPerName, v.getName(), v); //TODO scoping is not right
				}
			}
		}
		
		Set<String> result = new HashSet<String>();
		
		for (Map.Entry<String, Set<ASALVariable<?>>> entry : varsPerName.entrySet()) {
			//Variable names are automatically mCRL2 compatible.
			//Port names may have the ~ prefix, which should be removed:
			String name = getUnusedName(entry.getKey().replace("~", ""));
			result.add(name);
			
			for (ASALVariable<?> v : entry.getValue()) {
				setName(v, name);
			}
		}
		
		return result;
	}
	
	private Set<String> extractFuncNames() {
		Set<String> result = new HashSet<String>();
		
		//Function names need to be unique globally
		//(we use function names to look up the function body in a mapping):
		for (UnifyingBlock.ReprStateMachine rsm : target.reprStateMachines) {
			for (Map.Entry<String, ASALFunction> entry : rsm.representedStateMachine.functions.entrySet()) {
				String name = getUnusedName(entry.getKey());
				setName(entry.getValue(), name);
				result.add(name);
			}
		}
		
		return result;
	}
	
	private Set<String> extractStateNames() {
		Map<String, Set<TritoVertex>> verticesPerName = new HashMap<String, Set<TritoVertex>>();
		
		//State names only need to be unique for each state machine, but
		//nested states cannot have the same name as a composite state:
		for (UnifyingBlock.ReprStateMachine rsm : target.reprStateMachines) {
			UnusedNames unusedNames = new UnusedNames();
			
			for (TritoVertex v : rsm.representedStateMachine.vertices) {
				String name = unusedNames.generateUnusedName(v.getClz().getSimpleName());
				Set<TritoVertex> vertices = verticesPerName.get(name);
				
				if (vertices == null) {
					vertices = new HashSet<TritoVertex>();
					verticesPerName.put(name, vertices);
				}
				
				vertices.add(v);
			}
		}
		
		Set<String> result = new HashSet<String>();
		
		for (Map.Entry<String, Set<TritoVertex>> entry : verticesPerName.entrySet()) {
			//State names are automatically Java compatible:
			String name = getUnusedName(entry.getKey());
			result.add(name);
			
			for (TritoVertex v : entry.getValue()) {
				setName(v, name);
			}
		}
		
		return result;
	}
	
	private void extractTransitionNames() {
		for (UnifyingBlock.ReprStateMachine rsm : target.reprStateMachines) {
			UnusedNames unusedNames = new UnusedNames();
			
			for (TritoTransition t : rsm.representedStateMachine.transitions) {
				String name = unusedNames.generateUnusedName("T");
				setName(t, name);
			}
		}
	}
	
	private static String toFailsafeStr(String prefix, String s) {
		String result = prefix + "_";
		char prevChar = '_';
		
		for (int index = 0; index < s.length(); index++) {
			char c = s.charAt(index);
			
			if (Character.isJavaIdentifierPart(c)) {
				result += c;
				prevChar = c;
			} else {
				if (prevChar != '_') {
					result += "_";
				}
				
				result += Integer.toString((int)c);
				
				if (index + 1 < s.length()) {
					result += "_";
					prevChar = '_';
				} else {
					prevChar = c;
				}
			}
		}
		
		return result;
	}
	
	private Set<String> extractElemNames() {
		Set<String> result = new HashSet<String>();
		
		for (UnifyingBlock.ReprBlock rb : target.reprBlocks) {
			String rbName = getUnusedName(toFailsafeStr("comp", rb.name));
			setName(rb, rbName);
			result.add(rbName);
			
			String rbTypeName = getUnusedName(toFailsafeStr("CompState", rb.name.toUpperCase()));
			setName(rb.type, rbTypeName);
			result.add(rbTypeName);
		}
		
		return result;
	}
}




