package lib.sim;

import java.util.*;

import lib.asal.*;
import lib.asal.parsing.ASALException;
import lib.asal.parsing.api.*;
import lib.blocks.models.UnifyingBlock;
import lib.blocks.printing.AbstractPrinter;

public class ASAL2JavaSimVisitor extends ASALVisitor<List<String>> {
	public final AbstractPrinter<UnifyingBlock> printer;
	public final String componentName;
	
	public ASAL2JavaSimVisitor(AbstractPrinter<UnifyingBlock> printer, String componentName, ASALContext context) {
		super(context);
		
		this.printer = printer;
		this.componentName = componentName;
	}
	
	@Override
	public List<String> visitFctDef(ASALFunction fct) throws ASALException {
		List<String> result = new ArrayList<String>();
		result.add(printer.getName(fct));
		return result;
	}
	
	@Override
	public List<String> handle(ASALVariable<?> leaf) {
		List<String> result = new ArrayList<String>();
		
		switch (leaf.getOrigin()) {
			case FCT_PARAM:
				result.add(printer.getName(leaf));
				break;
			case STM_PORT:
			case STM_VAR:
				result.add(componentName + "." + printer.getName(leaf));
				break;
			default:
				break;
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALVarRef node, List<String> var) {
		List<String> result = new ArrayList<String>();
		
		switch (node.resolveVar(context).getOrigin()) {
			case FCT_PARAM:
				result.add(var.get(0));
				break;
			case STM_PORT:
			case STM_VAR:
				result.add(var.get(0) + ".value");
				break;
			default:
				break;
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALAssignStatement node, List<String> var, List<String> expr) {
		List<String> result = new ArrayList<String>();
		
		switch (node.resolveVar(context).getOrigin()) {
			case FCT_PARAM:
				result.add(var.get(0) + " = " + expr.get(0) + ";");
				break;
			case STM_PORT:
			case STM_VAR:
				result.add(var.get(0) + ".value = " + expr.get(0) + ";");
				break;
			default:
				break;
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALBinaryExpr node, List<String> lhs, List<String> rhs) {
		List<String> result = new ArrayList<String>();
		
		switch (node.getOp()) {
			case "+":
			case "-":
			case "*":
			case "/":
			case "%":
				result.add("Integer.valueOf(" + lhs.get(0) + " " + node.getOp() + " " +  rhs.get(0) + ")");
				break;
			case "and":
				result.add("Boolean.valueOf(" + lhs.get(0) + " && " +  rhs.get(0) + ")");
				break;
			case "or":
				result.add("Boolean.valueOf(" + lhs.get(0) + " || " +  rhs.get(0) + ")");
				break;
			case "=":
				result.add("Boolean.valueOf(" + lhs.get(0) + ".equals(" + rhs.get(0) + "))");
				break;
			case "<>":
				result.add("Boolean.valueOf(!" + lhs.get(0) + ".equals(" + rhs.get(0) + "))");
				break;
			case ">=":
			case "<=":
			case ">":
			case "<":
				result.add("Boolean.valueOf(" + lhs.get(0) + " " + node.getOp() + " " + rhs.get(0) + ")");
				break;
			default:
				throw new Error("Should not happen; no implementation for operator " + node.getOp() + "!");
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALEmptyStatement node) {
		return new ArrayList<String>();
	}
	
	@Override
	public List<String> handle(ASALFunction leaf, List<String> stat) {
		throw new Error("Should not happen!");
	}
	
	@Override
	public List<String> handle(ASALFunctionCall node, List<String> fct, Object... params) {
		String paramStr = "";
		
		if (params.length > 0) {
			paramStr = (String)params[0];
			
			for (int index = 1; index < params.length; index++) {
				paramStr += "," + (String)params[index];
			}
		}
		
		List<String> result = new ArrayList<String>();
		result.add(fct.get(0) + "(" + paramStr + ")");
		return result;
	}
	
	@Override
	public List<String> handle(ASALFunctionCallStatement node, List<String> fct, Object... params) {
		String paramStr = "";
		
		if (params.length > 0) {
			paramStr = (String)params[0];
			
			for (int index = 1; index < params.length; index++) {
				paramStr += "," + (String)params[index];
			}
		}
		
		List<String> result = new ArrayList<String>();
		result.add(fct.get(0) + "(" + paramStr + ");");
		return result;
	}
	
	@Override
	public List<String> handle(ASALIfStatement node, List<String> cond, List<String> thenBranch, List<String> elseBranch) {
		List<String> result = new ArrayList<String>();
		result.add("if (" + cond.get(0) + ") {");
		
		for (String s : thenBranch) {
			result.add("\t" + s);
		}
		
		if (elseBranch != null && elseBranch.size() > 0) {
			result.add("} else {");
			
			for (String s : elseBranch) {
				result.add("\t" + s);
			}
		}
		
		result.add("}");
		return result;
	}
	
	@Override
	public List<String> handle(ASALWhileStatement node, List<String> cond, List<String> body) {
		List<String> result = new ArrayList<String>();
		
		result.add("while (" + cond.get(0) + ") {");
		
		for (String s : body) {
			result.add("\t" + s);
		}
		
		result.add("}");
		return result;
	}
	
	@Override
	public List<String> handle(ASALLiteral leaf) {
		List<String> result = new ArrayList<String>();
		
		switch (leaf.getType(context)) {
			case BOOLEAN:
				result.add("Boolean." + leaf.getText());
				break;
			case NUMBER:
				result.add("Integer.valueOf(" + leaf.getText() + ")");
				break;
			case PULSE:
				result.add("Boolean." + leaf.getText());
				break;
			case STRING:
				result.add(leaf.getText());
				break;
			default:
				throw new Error("Should not happen!");
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALReturnStatement node, List<String> expr) {
		List<String> result = new ArrayList<String>();
		result.add("return " + expr.get(0) + ";");
		return result;
	}
	
	@Override
	public List<String> handle(ASALSeqStatement node, List<String> first, List<String> second) {
		List<String> result = new ArrayList<String>();
		result.addAll(first);
		result.addAll(second);
		return result;
	}
	
	@Override
	public List<String> handle(ASALUnaryExpr node, List<String> expr) {
		List<String> result = new ArrayList<String>();
		
		switch (node.getOp()) {
			case "+":
				result.add(expr.get(0));
				break;
			case "-":
				result.add("Integer.valueOf(-" + expr.get(0) + ")");
				break;
			case "not":
				result.add("Boolean.valueOf(!" + expr.get(0) + ")");
				break;
			default:
				throw new Error("Should not happen; no implementation for operator " + node.getOp() + "!");
		}
		
		return result;
	}
}




