package lib.blocks.printing;

import java.util.*;

import lib.blocks.common.*;
import lib.blocks.models.*;
import lib.blocks.models.IBDInstances.*;
import lib.blocks.models.ModelLib.Interface;
import lib.utils.*;

public class GraphVizModelPrinter2 extends AbstractPrinter<Model> {
	public GraphVizModelPrinter2(Model target) {
		super(target);
		
		target.initIBD2ConfProbs();
	}
	
	private String getConfProbLabel(Model.IBD2ConfProb prob) {
		return prob.exception.getCause().getMessage() + "\\n" + prob.constraint.toString();
	}
	
	@Override
	protected void print(String mode, Object object) {
		println("graph {");
		
		//Declare IBD1s:
		for (IBD1Instance ibd1 : target.getIBD1Instances()) {
			String elemName = getUnusedName(ibd1.id.toString());
			setName(ibd1, elemName);
			
			println("\t\"" + elemName + "\" [label=\"" + elemName + "\\n" + ibd1.getType().id.toString() + "\", shape=rect]");
		}
		
		for (IBD2Instance ibd2 : target.getIBD2Instances()) {
			String elemName = getUnusedName(ibd2.id.toString());
			setName(ibd2, elemName);
			
			println("\t\"" + elemName + "\" [label=\"" + elemName + "\\n" + ibd2.getType().id.toString() + "\", shape=rect, style=dashed]");
		}
		
		//Declare IBD2 ports:
		for (IBD2Port p : target.getIBD2Ports()) {
			String ptName = getUnusedName(p.id.toString());
			setName(p, ptName);
			
			Iterator<FlowSpecExpr> q = target.lib.getInterfaceExprs(p.someLegacyPt).iterator();
			String label;
			
			if (q.hasNext()) {
				label = q.next().toString();
				
				while (q.hasNext()) {
					label += ", " + q.next().toString();
				}
			} else {
				label = "???";
			}
			
			println("\t\"" + ptName + "\" [label=\"" + p.someLegacyPt.textify(LOD.MINIMAL) + ": " + label + "\\n(owned by " + p.owner.id + ": " + p.owner.getType().id + ")\", shape=oval, style=dotted]");
			
			for (Model.IBD2ConfProb prob : target.getIBD2ConfProbs(p)) {
				String probName = getUnusedName("prob");
				setName(prob, probName);
				
				println("\t\"" + probName + "\" [label=\"" + getConfProbLabel(prob) + "\", shape=rect, style=dashed, color=red, fontcolor=red]");
				println("\t\"" + ptName + "\" -- \"" + probName + "\" [label=\"\", dir=both, arrowtail=none, arrowhead=none, style=dashed, color=red]");
			}
		}
		
		Set<IBD1Port> exposedIBD1Ports = new HashSet<IBD1Port>(target.getIBD1Ports());
		Map<IBD2Port, Set<ModelLib.Field>> unusedFieldsPerIBD2Ports = new HashMap<IBDInstances.IBD2Port, Set<ModelLib.Field>>();
		Map<IBD2Port, Set<ModelLib.Field>> unusedConjFieldsPerIBD2Ports = new HashMap<IBDInstances.IBD2Port, Set<ModelLib.Field>>();
		
		for (IBD2Port p : target.getIBD2Ports()) {
			Set<ModelLib.Field> fields = new HashSet<ModelLib.Field>();
			
			for (FlowSpecExpr e : target.lib.getInterfaceExprs(p.someLegacyPt)) {
				Interface itf = target.lib.getInterface(e.flowSpecName);
				
				for (ModelLib.Field f : itf.getFields()) {
					fields.add(f);
				}
			}
			
			unusedFieldsPerIBD2Ports.put(p, new HashSet<ModelLib.Field>(fields));
			unusedConjFieldsPerIBD2Ports.put(p, new HashSet<ModelLib.Field>(fields));
		}
		
		Set<IBD2Port> ibd2PortsWithIBD1Flow = new HashSet<IBD2Port>();
		Set<IBD2Port> ibd2PortsWithIBD2Flow = new HashSet<IBD2Port>();
		
		//Draw lines between connected points:
		for (IBD1Flow f : target.getIBD1Flows()) {
			String name1 = getName(f.v1.owner);
			String name2 = getName(f.v2.owner);
			
			if (f.v1.dir == Dir.IN) {
				println("\t\"" + name1 + "\" -- \"" + name2 + "\" [label=\"\", dir=both, arrowtail=open, arrowhead=none, style=solid]");
			} else {
				// We only need to draw lines in one direction, because there is a flow from A to B and from B to A:
				// println("\t\"" + name1 + "\" -- \"" + name2 + "\" [label=\"\", dir=both, arrowtail=open, arrowhead=none, style=solid]");
			}
			
			exposedIBD1Ports.remove(f.v1);
			exposedIBD1Ports.remove(f.v2);
		}
		
		for (IBD2Flow f : target.getIBD2Flows()) {
			String name1 = getName(f.v1);
			String name2 = getName(f.v2);
			
			if (name1.compareTo(name2) < 0) {
				println("\t\"" + name1 + "\" -- \"" + name2 + "\" [label=\"\", dir=both, arrowtail=empty, arrowhead=empty, style=dashed]");
			}
			
			ibd2PortsWithIBD2Flow.add(f.v1);
			ibd2PortsWithIBD2Flow.add(f.v2);
		}
		
		for (IBD1ToIBD2Flow f : target.getIBD1ToIBD2Flows()) {
			String name1 = getName(f.v1.owner);
			String name2 = getName(f.v2);
			
			if (f.v1.dir == Dir.OUT) {
				println("\t\"" + name1 + "\" -- \"" + name2 + "\" [label=\"\", dir=both, arrowtail=none, arrowhead=open, style=solid]");
			} else {
				println("\t\"" + name1 + "\" -- \"" + name2 + "\" [label=\"\", dir=both, arrowtail=open, arrowhead=none, style=solid]");
			}
			
			exposedIBD1Ports.remove(f.v1);
			ibd2PortsWithIBD1Flow.add(f.v2);
		}
		
		Set<IBD2Port> exposedIBD2Ports = new HashSet<IBD2Port>();
		
		for (IBD2Port p : target.getIBD2Ports()) {
			boolean b1 = ibd2PortsWithIBD1Flow.contains(p);
			boolean b2 = ibd2PortsWithIBD2Flow.contains(p);
			
			if (!(b1 && b2)) {
				exposedIBD2Ports.add(p);
			}
		}
		
		Map<IBD1Instance, Set<IBD1Port>> exposedPortsPerIBD1Instance = new HashMap<IBDInstances.IBD1Instance, Set<IBD1Port>>();
		
		for (IBD1Instance ibd1Instance : target.getIBD1Instances()) {
			exposedPortsPerIBD1Instance.put(ibd1Instance, new HashSet<IBDInstances.IBD1Port>());
		}
		
		for (IBD1Port exposedPort : exposedIBD1Ports) {
			exposedPortsPerIBD1Instance.get(exposedPort.owner).add(exposedPort);
		}
		
		for (Map.Entry<IBD1Instance, Set<IBD1Port>> entry : exposedPortsPerIBD1Instance.entrySet()) {
			if (entry.getValue().size() > 0) {
				String exposedName = getUnusedName("exposed");
				String label = "EXPOSED PORT [" + entry.getValue().size() + "x]";
				
				for (IBD1Port p : entry.getValue()) {
					label += "\\n" + p.someLegacyPt.textify(LOD.FULL);
				}
				
				println("\t\"" + exposedName + "\" [label=\"" + label + "\", shape=rect, style=dashed, color=blue, fontcolor=blue]");
				println("\t\"" + getName(entry.getKey()) + "\" -- \"" + exposedName + "\" [label=\"\", dir=both, arrowtail=none, arrowhead=none, style=dashed, color=blue]");
			}
		}
		
		for (IBD2Port exposedPort : exposedIBD2Ports) {
			String exposedName = getUnusedName("exposed");
			String label = "EXPOSED INTERFACE [" + target.lib.getInterfaceExprs(exposedPort.someLegacyPt).size() + "x]";
			
			for (FlowSpecExpr e : target.lib.getInterfaceExprs(exposedPort.someLegacyPt)) {
				label += "\\n" + e.flowSpecName + (e.isConjugate ? " (~)" : "") + ":";
				Interface itf = target.lib.getInterface(e.flowSpecName);
				
				for (ModelLib.Field f : itf.getFields()) {
					label += "\\n" + f.toString(); 
				}
			}
			
			println("\t\"" + exposedName + "\" [label=\"" + label + "\", shape=rect, style=dashed, color=darkgreen, fontcolor=darkgreen]");
			println("\t\"" + getName(exposedPort) + "\" -- \"" + exposedName + "\" [label=\"\", dir=both, arrowtail=none, arrowhead=none, style=dashed, color=darkgreen]");
		}
		
		println("}");
	}
}
