package lib.blocks.printing;

import java.io.*;
import java.util.*;

import lib.utils.UnusedNames;

public abstract class AbstractPrinter<T> {
	private Map<Object, String> namePerObject;
	private UnusedNames unusedNames;
	private Stack<Mode> modes;
	
	private static class Mode {
		public final String name;
		public final List<PrintStream> out;
		public final Object object;
		
		private Mode(String name, Object object) {
			this.name = name;
			this.out = new ArrayList<PrintStream>();
			this.object = object;
		}
	}
	
	protected final T target;
	
	public AbstractPrinter(T target) {
		this.namePerObject = new HashMap<Object, String>();
		this.unusedNames = new UnusedNames();
		this.modes = new Stack<Mode>();
		this.target = target;
		
		pushMode("target", target);
	}
	
	public void pushMode(String name, Object object) {
		modes.push(new Mode(name, object));
	}
	
	public AbstractPrinter<T> add(String filename) {
		try {
			return add(new PrintStream(new File(filename)));
		} catch (FileNotFoundException e) {
			throw new Error(e);
		}
	}
	
	public AbstractPrinter<T> add(PrintStream o) {
		modes.peek().out.add(o);
		return this;
	}
	
	public AbstractPrinter<T> addSystemOut() {
		return add(System.out);
	}
	
	public final void setName(Object obj, String name) {
		namePerObject.put(obj, name);
	}
	
	public final String getName(Object obj) {
		return namePerObject.get(obj);
	}
	
	private int indentation = 0;
	
	protected void println(String s, int indent) {
		println(s);
		indentation += indent;
	}
	
	protected void println(int indent, String s) {
		indentation += indent;
		println(s);
	}
	
	protected void println(String s) {
		for (int index = 1; index <= indentation; index++) {
			s = "\t" + s;
		}
		
		for (PrintStream o : modes.peek().out) {
			o.println(s);
		}
	}
	
	protected final void println__(String s) {
		println("\t" + s);
	}
	
	protected final void println____(String s) {
		println("\t\t" + s);
	}
	
	protected final void println______(String s) {
		println("\t\t\t" + s);
	}
	
	protected final String getUnusedName(String base) {
		return unusedNames.generateUnusedName(base);
	}
	
	protected String applyNamingConventions(String base) {
		return base;
	}
	
	protected final UnusedNames cloneUnusedNames() {
		UnusedNames result = new UnusedNames();
		result.addAll(unusedNames);
		return result;
	}
	
	protected abstract void print(String mode, Object object);
	
	public final void printAndPop() {
		Mode mode = modes.peek();
		print(mode.name, mode.object);
		
		for (PrintStream o : modes.pop().out) {
			o.flush();
			o.close();
		}
	}
}
