package lib.blocks.printing;

import java.util.*;

import lib.asal.ASALDataType;
import lib.asal.ASALFunction;
import lib.asal.ASALVariable;
import lib.asal.parsing.ASALException;
import lib.asal.parsing.api.ASALLiteral;
import lib.behave.proto.*;
import lib.blocks.models.UnifyingBlock;
import lib.utils.UnusedNames;

public abstract class AbstractMCRL2Printer extends AbstractPrinter<UnifyingBlock> {
	protected final Set<String> varNames;
	protected final Set<String> funcNames;
	protected final Set<String> stateNames;
	protected final Set<String> stringValues;
	protected final Set<String> beqNames;
	
	protected final Map<String, String> customKeywords;
	
	public AbstractMCRL2Printer(UnifyingBlock target) throws ASALException {
		super(target);
		
		registerKeywords();
		customKeywords = getCustomKeywords();
		
		varNames = extractVarNames();
		funcNames = extractFuncNames();
		stateNames = extractStateNames();
		stringValues = extractStringValues();
		beqNames = extractElemNames();
	}
	
	private void registerKeywords() {
		getUnusedName("act");
		getUnusedName("allow");
		getUnusedName("block");
		getUnusedName("comm");
		getUnusedName("cons");
		getUnusedName("delay");
		getUnusedName("div");
		getUnusedName("end");
		getUnusedName("eqn");
		getUnusedName("exists");
		getUnusedName("forall");
		getUnusedName("glob");
		getUnusedName("hide");
		getUnusedName("if");
		getUnusedName("in");
		getUnusedName("init");
		getUnusedName("lambda");
		getUnusedName("map");
		getUnusedName("mod");
		getUnusedName("mu");
		getUnusedName("nu");
		getUnusedName("pbes");
		getUnusedName("proc");
		getUnusedName("rename");
		getUnusedName("sort");
		getUnusedName("struct");
		getUnusedName("sum");
		getUnusedName("val");
		getUnusedName("var");
		getUnusedName("whr");
		getUnusedName("yaled");
		getUnusedName("Bag");
		getUnusedName("Bool");
		getUnusedName("Int");
		getUnusedName("List");
		getUnusedName("Nat");
		getUnusedName("Pos");
		getUnusedName("Real");
		getUnusedName("Set");
		getUnusedName("delta");
		getUnusedName("false");
		getUnusedName("nil");
		getUnusedName("tau");
		getUnusedName("true");
		
		//TODO some keywords are missing, such as 'head' / 'rtail' / ...
	}
	
	@Override
	protected void println(String s) {
		String newS = s;
		
		for (Map.Entry<String, String> entry : customKeywords.entrySet()) {
			newS = newS.replace(entry.getKey(), entry.getValue());
		}
		
		super.println(newS);
	}
	
	private Map<String, String> getCustomKeywords() {
		Map<String, String> result = new HashMap<String, String>();
		
		//We reserve the following identifiers for our own goals:
		addCustomKeyword(result, "env", "Environment");
		addCustomKeyword(result, "value", "Value");
		addCustomKeyword(result, "int", "Value_Int");
		addCustomKeyword(result, "bool", "Value_Bool");
		addCustomKeyword(result, "pulse", "Value_Bool");
		addCustomKeyword(result, "code", "ASALCode");
		addCustomKeyword(result, "getCode", "getASALCode");
		addCustomKeyword(result, "isCode", "isASALCode");
		addCustomKeyword(result, "string", "Value_String");
		addCustomKeyword(result, "string.lit", "String");
		addCustomKeyword(result, "op1", "ASALUnaryOp");
		addCustomKeyword(result, "op1.+", "ASALUnaryOp_Plus");
		addCustomKeyword(result, "op1.-", "ASALUnaryOp_Minus");
		addCustomKeyword(result, "op1.not", "ASALUnaryOp_Negation");
		addCustomKeyword(result, "op2", "ASALBinaryOp");
		addCustomKeyword(result, "op2.+", "ASALBinaryOp_Add");
		addCustomKeyword(result, "op2.-", "ASALBinaryOp_Subtract");
		addCustomKeyword(result, "op2.*", "ASALBinaryOp_Mult");
		addCustomKeyword(result, "op2./", "ASALBinaryOp_Div");
		addCustomKeyword(result, "op2.%", "ASALBinaryOp_Mod");
		addCustomKeyword(result, "op2.=", "ASALBinaryOp_Eq");
		addCustomKeyword(result, "op2.<>", "ASALBinaryOp_Neq");
		addCustomKeyword(result, "op2.<=", "ASALBinaryOp_Leq");
		addCustomKeyword(result, "op2.>=", "ASALBinaryOp_Geq");
		addCustomKeyword(result, "op2.<", "ASALBinaryOp_Less");
		addCustomKeyword(result, "op2.>", "ASALBinaryOp_Greater");
		addCustomKeyword(result, "op2.and", "ASALBinaryOp_And");
		addCustomKeyword(result, "op2.or", "ASALBinaryOp_Or");
		addCustomKeyword(result, "op2.xor", "ASALBinaryOp_Xor");
		
		addCustomKeyword(result, "asala.pushGlobalVar", "ASALA_PushGlobalVar");
		addCustomKeyword(result, "asala.setGlobalVar", "ASALA_SetGlobalVar");
		addCustomKeyword(result, "asala.pushLocalVar", "ASALA_PushLocalVar");
		addCustomKeyword(result, "asala.setLocalVar", "ASALA_SetLocalVar");
		addCustomKeyword(result, "asala.op1", "ASALA_Op1");
		addCustomKeyword(result, "asala.op2", "ASALA_Op2");
		addCustomKeyword(result, "asala.fct", "ASALA_Fct");
		addCustomKeyword(result, "asala.jump", "ASALA_Jump");
		addCustomKeyword(result, "asala.jumpIfFalse", "ASALA_JumpIfFalse");
		addCustomKeyword(result, "asala.pushValue", "ASALA_PushValue");
		addCustomKeyword(result, "asala.return", "ASALA_Return");
		addCustomKeyword(result, "asala.pop", "ASALA_Pop");
		addCustomKeyword(result, "asala.pause", "ASALA_Pause");
		
		addCustomKeyword(result, "expr", "ASALExpr");
		addCustomKeyword(result, "expr.lit", "AExpr_Lit");
		addCustomKeyword(result, "expr.getPort", "AExpr_PortRef");
		addCustomKeyword(result, "expr.getParam", "AExpr_ParamRef");
		addCustomKeyword(result, "expr.op1", "AExpr_Unary");
		addCustomKeyword(result, "expr.op2", "AExpr_Binary");
		addCustomKeyword(result, "expr.fct", "AExpr_FunctionCall");
		addCustomKeyword(result, "stat", "ASALStat");
		addCustomKeyword(result, "stat.empty", "ASALStat_Empty");
		addCustomKeyword(result, "stat.setPort", "AStat_AssignPort");
		addCustomKeyword(result, "stat.setParam", "AStat_AssignParam");
		addCustomKeyword(result, "stat.fct", "AStat_FunctionCall");
		addCustomKeyword(result, "stat.if", "AStat_If");
		addCustomKeyword(result, "stat.while", "AStat_While");
		addCustomKeyword(result, "stat.return", "AStat_Return");
		addCustomKeyword(result, "stat.seq", "AStat_Seq");
		addCustomKeyword(result, "reg.key", "ASALRegEntry");
		addCustomKeyword(result, "reg.exitFuncFlag", "ASALExitFuncFlag");
		addCustomKeyword(result, "reg.pending", "ASALPendingCode");
		addCustomKeyword(result, "reg.top", "ASALStackTop");
		addCustomKeyword(result, "reg.port", "ASALPort");
		addCustomKeyword(result, "reg.param", "ASALParam");
		addCustomKeyword(result, "reg", "ASALReg");
		addCustomKeyword(result, "reg.init", "InitASALReg");
		addCustomKeyword(result, "reg.isDone", "isASALRegDone");
		addCustomKeyword(result, "reg.resume", "resumeASALRegCode");
		addCustomKeyword(result, "applyOp1", "applyUnaryOp");
		addCustomKeyword(result, "applyOp2", "applyBinaryOp");
		addCustomKeyword(result, "applyFct", "applyASALFunc");
		addCustomKeyword(result, "getPreFctReg", "getPreFctReg");
		addCustomKeyword(result, "evalExpr", "evalASALExpr");
		addCustomKeyword(result, "evalStat", "evalASALStat");
		
		addCustomKeyword(result, "port_channel");
		addCustomKeyword(result, "port_channels");
		addCustomKeyword(result, "pc.src.getComp", "port_channel_sender");
		addCustomKeyword(result, "pc.src.getPort", "port_channel_send_port");
		addCustomKeyword(result, "pc.tgt.getComp", "port_channel_receiver");
		addCustomKeyword(result, "pc.tgt.getPort", "port_channel_rcv_port");
		
		addCustomKeyword(result, "VarName");
		addCustomKeyword(result, "FunctionName");
		addCustomKeyword(result, "StateName");
		addCustomKeyword(result, "ComponentName");
		addCustomKeyword(result, "Event");
		
		addCustomKeyword(result, "func_params", "getFunctionParams");
		addCustomKeyword(result, "func_bodies", "getFunctionBodies");
		
		addCustomKeyword(result, "sm_config", "StateConfig");
		
		addCustomKeyword(result, "transition", "Transition");
		addCustomKeyword(result, "transitions");
		
		addCustomKeyword(result, "entry_action");
		addCustomKeyword(result, "exit_action");
		addCustomKeyword(result, "do_actions");
		
		return result;
	}
	
	private void addCustomKeyword(Map<String, String> dest, String value) {
		addCustomKeyword(dest, value, value);
	}
	
	private void addCustomKeyword(Map<String, String> dest, String key, String value) {
		if (dest.containsKey(key)) {
			throw new Error("Duplicate custom keyword entry: " + key);
		}
		
		dest.put("{{{" + key + "}}}", getUnusedName(value));
	}
	
	private static void addVarName(Map<String, Set<ASALVariable<?>>> varsPerName, String varName, ASALVariable<?> var) {
		Set<ASALVariable<?>> vars = varsPerName.get(varName);
		
		if (vars == null) {
			vars = new HashSet<ASALVariable<?>>();
			varsPerName.put(varName, vars);
		}
		
		vars.add(var);
	}
	
	private Set<String> extractVarNames() {
		Map<String, Set<ASALVariable<?>>> varsPerName = new HashMap<String, Set<ASALVariable<?>>>();
		
		for (UnifyingBlock.ReprBlock rb : target.reprBlocks) {
			for (UnifyingBlock.ReprPort rp : rb.ownedPorts) {
				addVarName(varsPerName, rp.getName(), rp);
			}
			
			if (rb.getOwnedStateMachine() != null) {
				UnifyingBlock.ReprStateMachine rsm =  rb.getOwnedStateMachine();
				
				for (Map.Entry<String, ASALVariable<?>> entry : rsm.representedStateMachine.stateMachineVars.entrySet()) {
					addVarName(varsPerName, entry.getKey(), entry.getValue());
				}
				
				//Ports should have already been added, but
				//they have a separate state machine declaration that we want to reference with the same id:
				for (Map.Entry<String, ASALVariable<?>> entry : rsm.representedStateMachine.inPortVars.entrySet()) {
					addVarName(varsPerName, entry.getKey(), entry.getValue());
				}
				
				for (Map.Entry<String, ASALVariable<?>> entry : rsm.representedStateMachine.outPortVars.entrySet()) {
					addVarName(varsPerName, entry.getKey(), entry.getValue());
				}
			}
		}
		
		Set<String> result = new HashSet<String>();
		
		for (Map.Entry<String, Set<ASALVariable<?>>> entry : varsPerName.entrySet()) {
			//Variable names are automatically mCRL2 compatible.
			//Port names may have the ~ prefix, which should be removed:
			String name = getUnusedName(entry.getKey().replace("~", ""));
			result.add(name);
			
			for (ASALVariable<?> v : entry.getValue()) {
				setName(v, name);
			}
		}
		
		return result;
	}
	
	private Set<String> extractFuncNames() {
		Set<String> result = new HashSet<String>();
		
		//Function names need to be unique globally
		//(we use function names to look up the function body in a mapping):
		for (UnifyingBlock.ReprStateMachine rsm : target.reprStateMachines) {
			for (Map.Entry<String, ASALFunction> entry : rsm.representedStateMachine.functions.entrySet()) {
				String name = getUnusedName(rsm.representedStateMachine.clz.getSimpleName() + "_" + entry.getKey());
				setName(entry.getValue(), name);
				result.add(name);
			}
		}
		
		return result;
	}
	
	private Set<String> extractStateNames() {
		Map<String, Set<TritoVertex>> verticesPerName = new HashMap<String, Set<TritoVertex>>();
		
		//State names only need to be unique for each state machine, but
		//nested states cannot have the same name as a composite state:
		for (UnifyingBlock.ReprStateMachine rsm : target.reprStateMachines) {
			UnusedNames unusedNames = new UnusedNames();
			
			for (TritoVertex v : rsm.representedStateMachine.vertices) {
				String name = unusedNames.generateUnusedName(v.getClz().getSimpleName());
				Set<TritoVertex> vertices = verticesPerName.get(name);
				
				if (vertices == null) {
					vertices = new HashSet<TritoVertex>();
					verticesPerName.put(name, vertices);
				}
				
				vertices.add(v);
			}
		}
		
		Set<String> result = new HashSet<String>();
		
		for (Map.Entry<String, Set<TritoVertex>> entry : verticesPerName.entrySet()) {
			//State names are automatically mCRL2 compatible:
			String name = getUnusedName(entry.getKey());
			result.add(name);
			
			for (TritoVertex v : entry.getValue()) {
				setName(v, name);
				//@ensures getName(v).equals(name);
			}
		}
		
		return result;
	}                  
	
	private Set<String> extractStringValues() throws ASALException {
		Set<ASALLiteral> stringLits = new HashSet<ASALLiteral>();
		
		for (UnifyingBlock.ReprStateMachine rsm : target.reprStateMachines) {
			StringLitCollectionVisitor v = new StringLitCollectionVisitor(rsm.representedStateMachine, stringLits);
			v.visitMachine(rsm.representedStateMachine);
		}
		
		Map<String, Set<ASALLiteral>> litsPerName = new HashMap<String, Set<ASALLiteral>>();
		
		for (ASALLiteral stringLit : stringLits) {
			String name = toFailsafeStr(stringLit);
			Set<ASALLiteral> lits = litsPerName.get(name);
			
			if (lits == null) {
				lits = new HashSet<ASALLiteral>();
				litsPerName.put(name, lits);
			}
			
			lits.add(stringLit);
		}
		
		for(List<ASALLiteral> litList: target.getEnvRestrictions().values()) {
			for(ASALLiteral lit : litList) {
				if(lit.getType() == ASALDataType.STRING) {
					String name = toFailsafeStr(lit);
					Set<ASALLiteral> lits = litsPerName.get(name);
					
					if (lits == null) {
						lits = new HashSet<ASALLiteral>();
						litsPerName.put(name, lits);
					}
					
					lits.add(lit);
				}
			}
		}
		
		for(ASALLiteral lit: target.getInitValuesEnvPorts().values()) {
			if(lit.getType() == ASALDataType.STRING) {
				String name = toFailsafeStr(lit);
				Set<ASALLiteral> lits = litsPerName.get(name);
				
				if (lits == null) {
					lits = new HashSet<ASALLiteral>();
					litsPerName.put(name, lits);
				}
				
				lits.add(lit);
			}
		}
		
		Set<String> result = new HashSet<String>();
		
		for (Map.Entry<String, Set<ASALLiteral>> entry : litsPerName.entrySet()) {
			String uniqueName = getUnusedName(entry.getKey());
			result.add(uniqueName);
			
			for (ASALLiteral lit : entry.getValue()) {
				setName(lit, uniqueName);
			}
		}
		
		return result;
	}
	
	private static String toFailsafeStr(ASALLiteral stringLit) {
		return toFailsafeStr("STR", stringLit.getText().substring(1, stringLit.getText().length() - 1));
	}
	
	private static String toFailsafeStr(String prefix, String s) {
		String result = prefix + "_";
		char prevChar = '_';
		
		for (int index = 0; index < s.length(); index++) {
			char c = s.charAt(index);
			
			if (Character.isJavaIdentifierPart(c)) {
				result += c;
				prevChar = c;
			} else {
				if (prevChar != '_') {
					result += "_";
				}
				
				result += Integer.toString((int)c);
				
				if (index + 1 < s.length()) {
					result += "_";
					prevChar = '_';
				} else {
					prevChar = c;
				}
			}
		}
		
		return result;
	}
	
	private static class StringLitCollectionVisitor extends TritoVisitor<Object> {
		private Collection<ASALLiteral> dest;
		
		public StringLitCollectionVisitor(TritoStateMachine stateMachine, Collection<ASALLiteral> dest) throws ASALException {
			super(stateMachine);
			
			this.dest = dest;
		}
		
		@Override
		public Object handle(ASALLiteral leaf) {
			switch (leaf.getType(context)) {
				case STRING:
					dest.add(leaf);
					break;
				default:
					break;
			}
			
			return super.handle(leaf);
		}
		
		@Override
		public Object visitFctDef(ASALFunction fct) throws ASALException {
			return null;
		}
	}
	
	private Set<String> extractElemNames() {
		Set<String> result = new HashSet<String>();
		
		for (UnifyingBlock.ReprBlock rb : target.reprBlocks) {
			String rbName = getUnusedName(toFailsafeStr("BEQ", rb.name));
			setName(rb, rbName);
			result.add(rbName);
		}
		
		return result;
	}
}




