package lib.blocks.printing;

import java.util.*;

import lib.asal.*;
import lib.asal.parsing.ASALException;
import lib.asal.parsing.api.*;
import lib.blocks.models.UnifyingBlock;

public class ASALA2mCRL2Visitor extends ASALVisitor<List<String>> {
	public final AbstractPrinter<UnifyingBlock> printer;
	
	public ASALA2mCRL2Visitor(AbstractPrinter<UnifyingBlock> printer, ASALContext context) {
		super(context);
		
		this.printer = printer;
	}
	
	@Override
	public List<String> visitFctDef(ASALFunction fct) throws ASALException {
		List<String> result = new ArrayList<String>();
		result.add(printer.getName(fct));
		return result;
	}
	
	@Override
	public List<String> handle(ASALVariable<?> leaf) {
		List<String> result = new ArrayList<String>();
		result.add(printer.getName(leaf));
		return result;
	}
	
	@Override
	public List<String> handle(ASALVarRef node, List<String> var) {
		List<String> result = new ArrayList<String>();
		
		switch (node.resolveVar(context).getOrigin()) {
			case FCT_PARAM:
				result.add("{{{asala.pushLocalVar}}}(" + var.get(0) + ")");
				break;
			case STM_PORT:
			case STM_VAR:
				result.add("{{{asala.pushGlobalVar}}}(" + var.get(0) + ")");
				break;
			default:
				break;
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALAssignStatement node, List<String> var, List<String> expr) {
		List<String> result = new ArrayList<String>();
		result.addAll(expr);
		
		switch (node.resolveVar(context).getOrigin()) {
			case FCT_PARAM:
				result.add("{{{asala.setLocalVar}}}(" + var.get(0) + ")");
				break;
			case STM_PORT:
			case STM_VAR:
				result.add("{{{asala.setGlobalVar}}}(" + var.get(0) + ")");
				break;
			default:
				break;
		}
		
		result.add("{{{asala.pop}}}");
		return result;
	}
	
	@Override
	public List<String> handle(ASALBinaryExpr node, List<String> lhs, List<String> rhs) {
		List<String> result = new ArrayList<String>();
		result.addAll(lhs);
		result.addAll(rhs);
		result.add("{{{asala.op2}}}({{{op2." + node.getOp() + "}}})");
		return result;
	}
	
	@Override
	public List<String> handle(ASALEmptyStatement node) {
		return new ArrayList<String>();
	}
	
	@Override
	public List<String> handle(ASALFunction leaf, List<String> stat) {
		throw new Error("Should not happen!");
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public List<String> handle(ASALFunctionCall node, List<String> fct, Object... params) {
		List<String> result = new ArrayList<String>();
		
		for (Object param : params) {
			result.addAll((List<String>)param);
		}
		
		result.add("{{{asala.fct}}}(" + fct.get(0) + ")");
		return result;
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public List<String> handle(ASALFunctionCallStatement node, List<String> fct, Object... params) {
		List<String> result = new ArrayList<String>();
		
		for (Object param : params) {
			result.addAll((List<String>)param);
		}
		
		result.add("{{{asala.fct}}}(" + fct.get(0) + ")");
		
		if (node.resolveFunction(context).getReturnType() != ASALDataType.VOID) {
			result.add("{{{asala.pop}}}");
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALIfStatement node, List<String> cond, List<String> thenBranch, List<String> elseBranch) {
		List<String> result = new ArrayList<String>();
		result.addAll(cond);
		
		if (elseBranch != null) {
			//Move to the end of the THEN branch, then to a jump instruction, and then one further:
			int delta1 = thenBranch.size() + 1 + 1;
			
			//Move to the end of the ELSE branch, and then one further:
			int delta2 = elseBranch.size() + 1;
			
			result.add("{{{asala.jumpIfFalse}}}(" + delta1 + ")");
			result.addAll(thenBranch);
			result.add("{{{asala.jump}}}(" + delta2 + ")");
			result.addAll(elseBranch);
		} else {
			//Move to the end of the THEN branch, then jump one further:
			int delta1 = thenBranch.size() + 1;
			
			result.add("{{{asala.jumpIfFalse}}}(" + delta1 + ")");
			result.addAll(thenBranch);
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALWhileStatement node, List<String> cond, List<String> body) {
		List<String> result = new ArrayList<String>();
		
		//Move to the end of the body, then to a jump instruction, and then one further:
		int delta1 = body.size() + 1 + 1;
		
		//Move to the start of the body, then to a jump instruction, and then to the start of the condition:
		int delta2 = -body.size() - 1 - cond.size();
		
		result.addAll(cond);
		result.add("{{{asala.jumpIfFalse}}}(" + delta1 + ")");
		result.addAll(body);
		result.add("{{{asala.jump}}}(" + delta2 + ")");
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALLiteral leaf) {
		List<String> result = new ArrayList<String>();
		
		switch (leaf.getType(context)) {
			case BOOLEAN:
				result.add("{{{asala.pushValue}}}(Value_Bool(" + leaf.getText().toLowerCase() + "))");
				break;
			case NUMBER:
				result.add("{{{asala.pushValue}}}(Value_Int(" + leaf.getText() + "))");
				break;
			case PULSE:
				result.add("{{{asala.pushValue}}}(Value_Bool(" + leaf.getText() + "))");
				break;
			case STRING:
				result.add("{{{asala.pushValue}}}(Value_String(" + printer.getName(leaf) + "))");
				break;
			default:
				throw new Error("Should not happen!");
		}
		
		return result;
	}
	
	@Override
	public List<String> handle(ASALReturnStatement node, List<String> expr) {
		List<String> result = new ArrayList<String>();
		result.addAll(expr);
		result.add("{{{asala.return}}}");
		return result;
	}
	
	@Override
	public List<String> handle(ASALSeqStatement node, List<String> first, List<String> second) {
		List<String> result = new ArrayList<String>();
		result.addAll(first);
		result.addAll(second);
		return result;
	}
	
	@Override
	public List<String> handle(ASALUnaryExpr node, List<String> expr) {
		List<String> result = new ArrayList<String>();
		result.addAll(expr);
		result.add("{{{asala.op1}}}({{{op1." + node.getOp() + "}}})");
		return result;
	}
}




