package lib.blocks.models;

import java.util.*;

import lib.asal.*;
import lib.asal.parsing.api.ASALLiteral;
import lib.behave.proto.TritoStateMachine;
import lib.blocks.common.ConnectionPt;
import lib.blocks.models.IBDInstances.*;
import lib.utils.*;

public class UnifyingBlock {
	public static class ReprBlock implements Textifiable {
		private ReprStateMachine ownedStateMachine;
		
		public final String name;
		public final Block type;
		public final Set<ReprPort> ownedPorts;
		
		public ReprBlock(String name, Block type) {
			this.name = name;
			this.type = type;
			
			ownedPorts = new HashSet<ReprPort>();
			ownedStateMachine = null;
		}
		
		public ReprStateMachine getOwnedStateMachine() {
			return ownedStateMachine;
		}
		
		@Override
		public String textify(LOD lod) {
			String result = name;
			
			if (lod.includesType) {
				result += ": " + type.id.toString();
			}
			
			return result;
		}
	}
	
	public static class ReprStateMachine implements Textifiable {
		public final TritoStateMachine representedStateMachine;
		
		private ReprStateMachine(TritoStateMachine representedStateMachine) {
			this.representedStateMachine = representedStateMachine;
		}
		
		@Override
		public String textify(LOD lod) {
			return representedStateMachine.clz.getName();
		}
	}
	
	public static class ReprPort implements ASALVariable<Object>, Textifiable {
		public final String name;
		public final ReprBlock owner;
		public final ASALVariable<?> stateMachineVar;
		public final ConnectionPt.PrimitivePt someReprPt;
		
		public ReprPort(String name, ReprBlock owner, ASALVariable<?> stateMachineVar, Set<ConnectionPt.PrimitivePt> reprPts) {
			this.name = name;
			this.owner = owner;
			this.stateMachineVar = stateMachineVar;
			
			someReprPt = reprPts.iterator().next();
		}
		
		@Override
		public String textify(LOD lod) {
			return someReprPt.toString();
		}
		
		@Override
		public String getName() {
			return name;
		}
		
		@Override
		public ASALVarOrigin getOrigin() {
			return ASALVarOrigin.STM_PORT;
		}
		
		@Override
		public ASALDataType getType() {
			return someReprPt.getType();
		}
		
		public Dir getDir() {
			return someReprPt.getDir();
		}
	}
	
	/**
	 * A representative flow between two ports (it may have been defined multiple times by the original blocks).
	 */
	public static class ReprFlow {
		public final ReprPort source;
		public final ReprPort target;
		
		private ReprFlow(ReprPort source, ReprPort target) {
			this.source = source;
			this.target = target;
		}
	}
	
	/**
	 * A representative communication event from one port to a set of ports.
	 */
	public static class ReprCommEvt {
		public final ReprPort source;
		public final Set<ReprPort> targets;
		public final Set<ReprFlow> usedFlows;
		
		public ReprCommEvt(ReprPort source) {
			this.source = source;
			
			targets = new HashSet<ReprPort>();
			usedFlows = new HashSet<ReprFlow>();
		}
	}
	
	/**
	 * A representative connection between two blocks, summarizing all flows that connect those blocks.
	 */
	public static class ReprConn implements Textifiable {
		public final ReprBlock source;
		public final ReprBlock target;
		public final Set<ReprFlow> representedFlows;
		public final Set<ConnectionPt.PrimitivePt> representedSrcPorts;
		public final Set<ConnectionPt.PrimitivePt> representedTgtPorts;
		
		public ReprConn(ReprBlock source, ReprBlock target) {
			this.source = source;
			this.target = target;
			
			representedFlows = new HashSet<ReprFlow>();
			representedSrcPorts = new HashSet<ConnectionPt.PrimitivePt>();
			representedTgtPorts = new HashSet<ConnectionPt.PrimitivePt>();
		}
		
		@Override
		public String textify(LOD lod) {
			// return Textifiable.textifyUnique(representedSrcPorts, "\n", lod);
			return "TODO"; // TODO
		}
	}
	
	public final Model model;
	
	public final List<ReprBlock> reprBlocks;
	public final List<ReprPort> reprPorts;
	public final List<ReprFlow> reprFlows;
	public final List<ReprConn> reprConns;
	public final List<ReprCommEvt> reprCommEvts;
	
	public final List<ReprStateMachine> reprStateMachines;
	
	public final Map<String, ReprBlock> reprBlockPerName;
	public final Map<TritoStateMachine, ReprStateMachine> reprSMPerSM;
	public final Map<IBDInstances.IBD1Port, ReprPort> reprPortPerPort;
	
	private Map<ReprPort, List<ASALLiteral>> EnvRestrictions = new HashMap<ReprPort, List<ASALLiteral>>();
	private Map<ReprPort, ASALLiteral> EnvInitValues = new HashMap<ReprPort, ASALLiteral>();
	
	public UnifyingBlock(Model model) {
		this.model = model;
		
		// Create blocks from elements.
		// Elements must be parts.
		// And yes, 'elements' in the model are called 'blocks' here, just because:
		reprBlockPerName = new HashMap<String, ReprBlock>();
		reprBlocks = new ArrayList<ReprBlock>();
		
		reprSMPerSM = new HashMap<TritoStateMachine, ReprStateMachine>();
		reprStateMachines = new ArrayList<ReprStateMachine>();
		
		for (IBD1Instance e : model.getIBD1Instances()) {
			ReprBlock rb = new ReprBlock(e.id.toString(), e.getType());
			reprBlockPerName.put(rb.name, rb);
			reprBlocks.add(rb);
			
			if (e.getStateMachine() != null) {
				ReprStateMachine rsm = new ReprStateMachine(e.getStateMachine());
				reprSMPerSM.put(e.getStateMachine(), rsm);
				reprStateMachines.add(rsm);
				rb.ownedStateMachine = rsm;
			}
		}
		
		// Create ports and attach them to a block:
		reprPortPerPort = new HashMap<IBDInstances.IBD1Port, ReprPort>();
		reprPorts = new ArrayList<ReprPort>();
		
		for (IBD1Port e : model.getIBD1Ports()) {
			ReprBlock portOwner = reprBlockPerName.get(e.owner.id.toString());
			ASALVariable<?> stateMachineVar = null;
			
			if (portOwner.ownedStateMachine != null) {
				stateMachineVar = portOwner.ownedStateMachine.representedStateMachine.inPortVars.get(e.name);
				
				if (stateMachineVar == null) {
					stateMachineVar = portOwner.ownedStateMachine.representedStateMachine.outPortVars.get(e.name);
				}
			}
			
			ReprPort rp = new ReprPort(e.name, portOwner, stateMachineVar, e.getLegacyPts());
			portOwner.ownedPorts.add(rp);
			reprPorts.add(rp);
			
			reprPortPerPort.put(e, rp);
		}
		
		// Create flows + communication events:
		reprFlows = new ArrayList<ReprFlow>();
		reprCommEvts = new ArrayList<ReprCommEvt>();
		
		for (IBD1Port outPort : model.getIBD1Ports()) {
			//Flows only flow from output port to one or more input ports:
			if (outPort.someLegacyPt.getDir() == Dir.OUT) {
				ReprPort rp1 = reprPortPerPort.get(outPort);
				ReprCommEvt evt = new ReprCommEvt(rp1);
				
				for (IBD1Port inPort : model.getConjugateIBD1Ports(outPort)) {
					ReprPort rp2 = reprPortPerPort.get(inPort);
					ReprFlow flow = new ReprFlow(rp1, rp2);
					evt.targets.add(flow.target);
					evt.usedFlows.add(flow);
					reprFlows.add(flow);
				}
				
				if (evt.usedFlows.size() > 0) {
					reprCommEvts.add(evt);
				}
			}
		}
		
		// Summarize flows as flow `connections' (= all flows that connect two specific blocks):
		reprConns = new ArrayList<ReprConn>();
		
		for (ReprFlow f : reprFlows) {
			boolean added = false;
			
			for (ReprConn conn : reprConns) {
				if (conn.source == f.source.owner && conn.target == f.target.owner) {
					conn.representedFlows.add(f);
					conn.representedSrcPorts.add(f.source.someReprPt);
					conn.representedTgtPorts.add(f.target.someReprPt);
					added = true;
					break;
				}
			}
			
			if (!added) {
				ReprConn conn = new ReprConn(f.source.owner, f.target.owner);
				conn.representedFlows.add(f);
				conn.representedSrcPorts.add(f.source.someReprPt);
				conn.representedTgtPorts.add(f.target.someReprPt);
				reprConns.add(conn);
			}
		}
	}
	
	public boolean isPortToEnvironment(ReprPort port) {
		for (ReprFlow f : reprFlows) {
			if (f.source == port || f.target == port) {
				return false;
			}
		}
		
		return true;
	}
	
	public Map<ReprPort, List<ASALLiteral>> getEnvRestrictions() {
		return this.EnvRestrictions;
	}
	
	public Map<ReprPort, ASALLiteral> getInitValuesEnvPorts() {
		return this.EnvInitValues;
	}
	
	public void setEnvRestrictions(Map<IBDInstances.IBD1Port, List<ASALLiteral>> restr) {
		this.EnvRestrictions = new HashMap<ReprPort, List<ASALLiteral>>();
		
		for(Map.Entry<IBDInstances.IBD1Port, List<ASALLiteral>> entry : restr.entrySet()) {
			ReprPort reprPort = this.reprPortPerPort.get(entry.getKey());
			List<ASALLiteral> possibleValues = entry.getValue();
			
			if(!isPortToEnvironment(reprPort)) {
				throw new Error("Port " + reprPort.name + " is not a port connected to the environment, can't be restricted");
			}
			
			this.EnvRestrictions.put(reprPort, possibleValues);
		}
	}
	
	public void setInitialValuesEnvPorts(Map<IBDInstances.IBD1Port, ASALLiteral> init) {
		this.EnvInitValues = new HashMap<ReprPort, ASALLiteral>();
		
		for(Map.Entry<IBDInstances.IBD1Port, ASALLiteral> entry : init.entrySet()) {
			ReprPort reprPort = this.reprPortPerPort.get(entry.getKey());
			
			if(!isPortToEnvironment(reprPort)) {
				throw new Error("Port " + reprPort.name + " is not a port connected to the environment, can't be restricted");
			}
			
			this.EnvInitValues.put(reprPort, entry.getValue());
		}
	}
}
