package lib.blocks.models;

import java.lang.reflect.Field;
import java.util.*;

import lib.blocks.bdds.*;
import lib.blocks.ibd1.*;
import lib.blocks.ibd2.*;
import lib.utils.*;

public enum SourceType {
	/**
	 * Classes are in this category if and only if they inherit from {@link #BDD}.
	 */
	BDD("BDD"),
	
	/**
	 * Classes are in this category if and only if they inherit from {@link #IBD1Target}.
	 */
	IBD1("IBD1"),
	
	/**
	 * Classes are in this category if and only if they contain IBD2s.
	 */
	IBD2_INTERFACE("IBD2 interface"),
	
	/**
	 * Classes are in this category if and only if they inherit from {@link #IBD2Target}.
	 */
	IBD2("IBD2"),
	
	;
	
	public final String prettyName;
	
	private SourceType(String prettyName) {
		this.prettyName = prettyName;
	}
	
	/**
	 * If a user-defined specification class belongs to one of the SourceType categories,
	 * then the SourceType category can be established statically with this method.
	 */
	public static SourceType get(Class<?> clz) {
		if (clz.getSuperclass() != null) {
			if (clz.getDeclaringClass() != null && BDD.class.isAssignableFrom(clz.getDeclaringClass())) {
				return BDD;
			}
			
			if (IBD1Target.class.isAssignableFrom(clz)) {
				return IBD1;
			}
			
			if (IBD2Target.class.isAssignableFrom(clz)) {
				return IBD2;
			}
			
			if (IBD2Interface.class.isAssignableFrom(clz)) {
				return IBD2_INTERFACE;
			}
		}
		
		//Categorization failed:
		throw new Error (
			clz.getCanonicalName() + " fails to meet one of the following requirements:\n" +
			"	1. Inherit from " + IBD1Target.class.getCanonicalName() + ";\n" +
			"	2. Inherit from " + IBD2Target.class.getCanonicalName() + ";\n" +
			"	3. Inherit from " + IBD2Interface.class.getCanonicalName() + ";\n" +
			"	4. Inherit from " + Object.class.getCanonicalName() + " while nested inside a subclass of " + BDD.class.getCanonicalName() + "!\n"
		);
	}
	
	/**
	 * Obtains all types that are referenced by the given user-defined class.
	 * Includes the given user-defined class.
	 * BDD-types will only reference BDD-types, and
	 * IBD-types will only reference IBD-types.
	 */
	public static Set<Class<?>> getReferencedTypes(Class<?> clz) {
		Set<Class<?>> result = new HashSet<Class<?>>();
		addReferencedTypes(clz, result);
		return result;
	}
	
	public static void addReferencedTypes(Class<?> clz, Set<Class<?>> dest) {
		if (dest.add(clz)) {
			switch (get(clz)) {
				case BDD:
					for (Class<?> c : clz.getDeclaringClass().getDeclaredClasses()) {
						addReferencedTypes(c, dest);
					}
					break;
				case IBD1:
					if (clz.getSuperclass() != IBD1Target.class) {
						addReferencedTypes(clz.getSuperclass(), dest);
					}
					break;
				case IBD2:
					if (clz.getSuperclass() != IBD2Target.class) {
						addReferencedTypes(clz.getSuperclass(), dest);
					}
					
					if (clz.getDeclaringClass() != null) {
						addReferencedTypes(clz.getDeclaringClass(), dest);
					}
					
					for (Field f : ClassUtils.getAllFields(clz)) {
						Class<?> type;
						
						if (f.getType().isArray()) {
							type = f.getType().getComponentType();
						} else {
							type = f.getType();
						}
						
						if (IBD1Target.class.isAssignableFrom(type)) {
							addReferencedTypes(f.getType(), dest);
						}
						
						if (IBD2Target.class.isAssignableFrom(type)) {
							addReferencedTypes(f.getType(), dest);
						}
					}
					break;
				case IBD2_INTERFACE:
					if (clz.getSuperclass() != IBD2Interface.class) {
						addReferencedTypes(clz.getSuperclass(), dest);
					}
					
					for (Class<?> c : clz.getDeclaredClasses()) {
						addReferencedTypes(c, dest);
					}
					break;
			}
		}
	}
	
	/**
	 * Categorize the given class, and
	 * check the consistency of the class based its categorization.
	 * Also checks the consistency of referenced classes.
	 */
	public static void confirmConsistentSrc(Class<?> clz) throws ReflectionException {
		confirmConsistentSrc(clz, new HashSet<Class<?>>());
	}
	
	private static void confirmConsistentSrc(Class<?> clz, Set<Class<?>> beenHere) throws ReflectionException {
		if (beenHere.add(clz)) { //Avoid infinite recursion.
			switch (get(clz)) {
				case BDD:
					ClassUtils.confirmUnnested(clz.getDeclaringClass());
					ClassUtils.confirmStdClass(clz.getDeclaringClass(), ModReq.NOT_STATIC);
					ClassUtils.confirmSuperclass(clz.getDeclaringClass(), BDD.class);
					ClassUtils.confirmInterfaces(clz.getDeclaringClass());
					
					Set<Class<?>> peerClasses = new HashSet<Class<?>>();
					
					for (Class<?> peerClz : clz.getDeclaringClass().getDeclaredClasses()) {
						peerClasses.add(peerClz);
					}
					
					ClassUtils.confirmStdClass(clz, ModReq.STATIC);
					ClassUtils.confirmNoNestedClasses(clz);
					ClassUtils.confirmSuperclass(clz, peerClasses, Object.class);
					ClassUtils.confirmInterfaces(clz); //No interfaces!
					ClassUtils.confirmStdFields(clz, ModReq.STATIC, Relation.class, String.class);
					ClassUtils.confirmAssignedFields(clz, ModReq.ASSIGNED);
					ClassUtils.getFieldValue(clz, "REFERENCE", String.class, null);
					break;
				case IBD1:
					ClassUtils.confirmUnnested(clz);
					ClassUtils.confirmStdClass(clz, ModReq.NOT_STATIC);
					ClassUtils.confirmNoNestedClasses(clz);
					ClassUtils.confirmInheritsFrom(clz, IBD1Target.class);
					ClassUtils.confirmInterfaces(clz); //No interfaces!
					ClassUtils.confirmStdFields(clz, ModReq.NOT_STATIC, InPort.class, OutPort.class, String.class);
					ClassUtils.confirmAssignedFields(clz, ModReq.ASSIGNED);
					ClassUtils.getFieldValue(clz, "REFERENCE", String.class, null);
					break;
				case IBD2:
					if (clz.getDeclaringClass() != null) {
						ClassUtils.confirmStdClass(clz, ModReq.STATIC);
						ClassUtils.confirmSuperclass(clz, Object.class);
					} else {
						ClassUtils.confirmUnnested(clz);
						ClassUtils.confirmStdClass(clz, ModReq.NOT_STATIC);
					}
					
					ClassUtils.confirmInheritsFrom(clz, IBD2Target.class);
					ClassUtils.confirmInterfaces(clz); //No interfaces!
					ClassUtils.confirmAssignedFields(clz, ModReq.ASSIGNED);
					ClassUtils.confirmStdFields(clz, ModReq.NOT_STATIC, (x) -> {
						if (x.isArray()) {
							Class<?> compType = x.getComponentType();
							
							if (IBD1Target.class.isAssignableFrom(compType)) {
								return true;
							}
							
							if (IBD2Target.class.isAssignableFrom(compType)) {
								return true;
							}
							
							if (InterfacePort.class.isAssignableFrom(compType)) {
								return true;
							}
						}
						
						return false;
					}, IBD1Target.class, IBD2Target.class, InterfacePort.class, String.class);
					ClassUtils.getFieldValue(clz, "REFERENCE", String.class, null);
					break;
				case IBD2_INTERFACE:
					ClassUtils.confirmUnnested(clz.getDeclaringClass());
					ClassUtils.confirmStdClass(clz.getDeclaringClass(), ModReq.NOT_STATIC);
					ClassUtils.confirmSuperclass(clz, Object.class);
					ClassUtils.confirmInterfaces(clz); //No interfaces!
					ClassUtils.confirmStdFields(clz, ModReq.STATIC, String.class);
					ClassUtils.getFieldValue(clz, "REFERENCE", String.class, null);
					
					for (Class<?> c : ClassUtils.getDeclaredClasses(clz)) {
						if (SourceType.get(c) == SourceType.IBD2) {
							confirmConsistentSrc(c);
						} else {
							throw new Error("\"" + c.getCanonicalName() + "\" should be an IBD2!");
						}
					}
					break;
			}
		}
	}
	
	private static void categorize(Class<?> userDefinedClz, Map<SourceType, Set<Class<?>>> dest) {
		SourceType sourceType = get(userDefinedClz);
		Set<Class<?>> userDefinedClzs = dest.get(sourceType);
		
		if (userDefinedClzs == null) {
			userDefinedClzs = new HashSet<Class<?>>();
			dest.put(sourceType, userDefinedClzs);
		}
		
		userDefinedClzs.add(userDefinedClz);
	}
	
	public static Map<SourceType, Set<Class<?>>> categorizeUserDefinedClzs(Block block) {
		Map<SourceType, Set<Class<?>>> result = new HashMap<SourceType, Set<Class<?>>>();
		
		for (Block b : block.getLineage()) {
			for (Class<?> clz : b.getUserDefinedClzs()) {
				categorize(clz, result);
			}
		}
		
		return result;
	}
}
