package lib.blocks.models;

import java.util.*;

/**
 * Helper class for the construction of a bijection from name-index-ids to name-index-ids.
 */
public class NameIndexIdMap {
	private Map<NameIndexId, NameIndexId> oldIdPerNewId;
	private Map<NameIndexId, NameIndexId> newIdPerOldId;
	
	public NameIndexIdMap() {
		oldIdPerNewId = new HashMap<NameIndexId, NameIndexId>();
		newIdPerOldId = new HashMap<NameIndexId, NameIndexId>();
	}
	
	public void clear() {
		oldIdPerNewId.clear();
		newIdPerOldId.clear();
	}
	
	public void add(NameIndexId oldId, NameIndexId newId) {
		NameIndexId existingId = newIdPerOldId.get(oldId);
		
		if (existingId != null) {
			if (!existingId.equals(newId)) {
				throw new Error("Behavior of renaming scheme should be deterministic!");
			}
		}
		
		existingId = oldIdPerNewId.get(newId);
		
		if (existingId != null) {
			if (!existingId.equals(oldId)) {
				throw new Error("Cannot rename both " + oldId + " and " + existingId + " to " + newId + "!");
			}
		}
		
		newIdPerOldId.put(oldId, newId);
		oldIdPerNewId.put(newId, oldId);
	}
	
	public NameIndexId get(NameIndexId oldId) {
		if (!newIdPerOldId.containsKey(oldId)) {
			String msg = "No value found for key " + oldId + "; mapping contains the following:";
			
			for (Map.Entry<NameIndexId, NameIndexId> entry : newIdPerOldId.entrySet()) {
				msg += "\n\t" + entry.getKey() + " -> " + entry.getValue();
			}
			
			throw new Error(msg);
		}
		
		return newIdPerOldId.get(oldId);
	}
	
	public NameIndexId getInversed(NameIndexId newId) {
		if (!oldIdPerNewId.containsKey(newId)) {
			String msg = "No key found for value " + newId + "; mapping contains the following:";
			
			for (Map.Entry<NameIndexId, NameIndexId> entry : oldIdPerNewId.entrySet()) {
				msg += "\n\t" + entry.getKey() + " -> " + entry.getValue();
			}
			
			throw new Error(msg);
		}
		
		return oldIdPerNewId.get(newId);
	}
	
	public Set<NameIndexId> keySet() {
		return Collections.unmodifiableSet(newIdPerOldId.keySet());
	}
	
	public Set<NameIndexId> values() {
		return Collections.unmodifiableSet(oldIdPerNewId.keySet());
	}
}
