package lib.blocks.models;

import java.util.*;

import lib.asal.ASALDataType;
import lib.blocks.bdds.BDD;
import lib.blocks.common.*;
import lib.blocks.constraints.*;
import lib.blocks.ibd2.IBD2Target;
//import lib.blocks.printing.GraphVizModelPrinter;
import lib.utils.*;

public class ModelLib {
	private Map<Class<?>, Block> blockPerUserDefinedClz;
	private Map<BlockId, Block> blockPerId;
	private Map<Block, BlockCache> cachePerBlock;
	
	private Map<String, Interface> flowSpecPerName;
	private Map<ConnectionPt.InterfacePt, Set<FlowSpecExpr>> flowSpecsPerOwnedPt;
	private Map<ConnectionPt.PrimitivePt, Set<NameIndexId>> interfaceNamesPerPrimPt;
	
	public static class BlockCache {
		private Block block;
		private Block superBlock;
		private boolean isSuperBlockAssigned;
		
		private SourceType sourceType;
		private Set<Class<?>> userDefinedClzs;
		private List<BlockConstraint> declaredConstraints;
		private List<BlockConstraint> inheritedConstraints;
		private Set<ConnectionPt> contextPts;
		private Set<ConnectionPt> ownedPts;
		
		public BlockCache(Block block) {
			this.block = block;
			
			sourceType = null;
			isSuperBlockAssigned = false;
			userDefinedClzs = new HashSet<Class<?>>();
			declaredConstraints = new ArrayList<BlockConstraint>();
			inheritedConstraints = new ArrayList<BlockConstraint>();
			contextPts = new HashSet<ConnectionPt>();
			ownedPts = new HashSet<ConnectionPt>();
		}
		
		public Block getBlock() {
			return block;
		}
		
		private void clearProcessed() {
			superBlock = null;
			isSuperBlockAssigned = false;
			sourceType = null;
			declaredConstraints.clear();
			inheritedConstraints.clear();
			contextPts.clear();
			ownedPts.clear();
		}
		
		public SourceType getSourceType() {
			return sourceType;
		}
		
		public Block getSuperBlock() {
			if (!isSuperBlockAssigned) {
				isSuperBlockAssigned = true;
				superBlock = null;
				
				for (BlockConstraint constraint : declaredConstraints) {
					if (constraint instanceof GeneralizationConstraint) {
						GeneralizationConstraint a = (GeneralizationConstraint)constraint;
						superBlock = a.getSuperBlock();
						break;
					}
				}
			}
			
			return superBlock;
		}
		
		public Set<Class<?>> getUserDefinedClzs() {
			return Collections.unmodifiableSet(userDefinedClzs);
		}
		
		public List<BlockConstraint> getDeclaredConstraints() {
			return Collections.unmodifiableList(declaredConstraints);
		}
		
		public List<BlockConstraint> getInheritedConstraints() {
			return Collections.unmodifiableList(inheritedConstraints);
		}
		
		/**
		 * Declared AND inherited!
		 */
		public Set<ConnectionPt> getContextPts() {
			return Collections.unmodifiableSet(contextPts);
		}
		
		/**
		 * Not inherited!
		 */
		public Set<ConnectionPt> getOwnedPts() {
			return Collections.unmodifiableSet(ownedPts);
		}
	}
	
	public static class Field {
		public final Set<ConnectionPt> contributors;
		public final String suffix;
		public final ASALDataType type;
		public final Dir dir;
		
		private Field(ConnectionPt contributor, String suffix, ASALDataType type, Dir dir) {
			this.suffix = suffix;
			this.type = type;
			this.dir = dir;
			
			contributors = new HashSet<ConnectionPt>();
			contributors.add(contributor);
		}
		
		public boolean isSame(Field other) {
			return suffix.equals(other.suffix) && type == other.type && dir == other.dir;
		}
		
		@Override
		public String toString() {
			return dir.text + " " + suffix + ": " + type.name;
		}
	}
	
	public static class Interface {
		private Map<String, Field> fieldPerSuffix;
		
		public final String name;
		
		public Interface(String name) {
			this.name = name;
			
			fieldPerSuffix = new HashMap<String, Field>();
		}
		
		public Set<String> getSuffixes() {
			return Collections.unmodifiableSet(fieldPerSuffix.keySet());
		}
		
		public Collection<Field> getFields() {
			return Collections.unmodifiableCollection(fieldPerSuffix.values());
		}
		
		public Field getField(String suffix) {
			Field result = fieldPerSuffix.get(suffix);
			
			if (result == null) {
				String msg = "Unknown field suffix \"" + name + "\"! Known field suffixes:";
				
				for (Field field : fieldPerSuffix.values()) {
					msg += "\n\t" + field.suffix;
				}
				
				throw new Error(msg);
			}
			
			return result;
		}
		
		private void addField(Field newField) {
			Field existingField = fieldPerSuffix.get(newField.suffix);
			
			if (existingField != null) {
				if (newField.isSame(existingField)) {
					existingField.contributors.addAll(newField.contributors);
				} else {
					String msg = "Incompatible contributions to interface \"" + name + "\":";
					msg += "\n\t" + existingField.toString();
					
					for (ConnectionPt pt : existingField.contributors) {
						msg += "\n\t\t" + pt.toString();
					}
					
					msg += "\n\t" + newField.toString();
					
					for (ConnectionPt pt : newField.contributors) {
						msg += "\n\t\t" + pt.toString();
					}
					
					throw new Error(msg);
				}
			} else {
				fieldPerSuffix.put(newField.suffix, newField);
			}
		}
		
		public void print() {
			System.out.println("interface " + name + " {");
			
			for (Map.Entry<String, Field> entry : fieldPerSuffix.entrySet()) {
				System.out.println("\t" + entry.getValue().dir.text + " " + entry.getKey() + ": " + entry.getValue().type.javaClass.getSimpleName());
			}
			
			System.out.println("}");
		}
	}
	
	public ModelLib() {
		blockPerUserDefinedClz = new HashMap<Class<?>, Block>();
		blockPerId = new HashMap<BlockId, Block>();
		cachePerBlock = new HashMap<Block, ModelLib.BlockCache>();
		
		flowSpecPerName = new HashMap<String, Interface>();
		flowSpecsPerOwnedPt = new HashMap<ConnectionPt.InterfacePt, Set<FlowSpecExpr>>();
		interfaceNamesPerPrimPt = new HashMap<ConnectionPt.PrimitivePt, Set<NameIndexId>>();
	}
	
	public void clear() {
		blockPerUserDefinedClz.clear();
		blockPerId.clear();
		cachePerBlock.clear();
		
		flowSpecPerName.clear();
		flowSpecsPerOwnedPt.clear();
		interfaceNamesPerPrimPt.clear();
	}
	
	/**
	 * Names of all declared blocks.
	 * Note that block <i>variants</i> use the name of their declaring class.
	 */
	public Set<BlockId> getBlockIds() {
		return Collections.unmodifiableSet(blockPerId.keySet());
	}
	
	public Collection<Block> getBlocks() {
		return Collections.unmodifiableCollection(blockPerId.values());
	}
	
	public Set<String> getInterfaceNames() {
		return Collections.unmodifiableSet(flowSpecPerName.keySet());
	}
	
	public Collection<Interface> getInterfaces() {
		return Collections.unmodifiableCollection(flowSpecPerName.values());
	}
	
	public Interface getInterface(String name) {
		Interface result = flowSpecPerName.get(name);
		
		if (result == null) {
			String msg = "Unknown interface \"" + name + "\"! Known interfaces:";
			
			for (Interface itf : flowSpecPerName.values()) {
				msg += "\n\t" + itf.name;
			}
			
			throw new Error(msg);
		}
		
		return result;
	}
	
	public Set<FlowSpecExpr> getInterfaceExprs(ConnectionPt.InterfacePt ownedPt) {
		Set<FlowSpecExpr> result = flowSpecsPerOwnedPt.get(ownedPt);
		
		if (result == null) {
			throw new Error("Should not happen; does not have interface expressions:\n\t" + ownedPt.toString());
		}
		
		return result;
	}
	
	public Set<NameIndexId> getInterfaceNames(ConnectionPt.PrimitivePt primPt) {
		Set<NameIndexId> result = interfaceNamesPerPrimPt.get(primPt);
		
		if (result == null) {
			throw new Error("Should not happen; does not have interface names:\n\t" + primPt.toString());
		}
		
		return result;
	}
	
	public void add(Class<?> clz) throws ReflectionException {
		if (clz.getDeclaringClass() != null) {
			throw new Error("Please do not add diagram elements directly!");
		}
		
		if (clz.getSuperclass() == BDD.class) {
			for (Class<?> c : clz.getDeclaredClasses()) {
				SourceType.confirmConsistentSrc(c);
				getOrCreateBlock(c);
			}
		} else {
			SourceType.confirmConsistentSrc(clz);
			getOrCreateBlock(clz);
		}
	}
	
	private BlockCache getOrCreateBlock(Class<?> clz) throws ReflectionException {
		SourceType.confirmConsistentSrc(clz);
		
		BlockId blockId = new BlockId(clz);
		Block block = blockPerId.get(blockId);
		
		if (block == null) {
			block = new Block(this, blockId);
			blockPerId.put(blockId, block);
		}
		
		BlockCache cache = cachePerBlock.get(block);
		
		if (cache == null) {
			cache = new BlockCache(block);
			cachePerBlock.put(block, cache);
		}
		
		cache.userDefinedClzs.add(clz);
		blockPerUserDefinedClz.put(clz, block);
		return cache;
	}
	
	public Block get(Class<?> clz) {
		Block result = blockPerUserDefinedClz.get(clz);
		
		if (result == null) {
			String msg = "Unknown block \"" + clz.getCanonicalName() + "\"! Known blocks:";
			
			for (Block block : blockPerUserDefinedClz.values()) {
				msg += "\n\t" + block.id;
			}
			
			throw new Error(msg);
		}
		
		return result;
	}
	
	public BlockCache get(Block block) {
		return cachePerBlock.get(block);
	}
	
	public void declareConstraint(Class<?> targetBlockClz, BlockConstraint constraint) throws IncompatibilityException {
		insertBlockConstraint(constraint, get(get(targetBlockClz)).declaredConstraints, false);
	}
	
	public void declareConstraint(Block targetBlock, BlockConstraint constraint) throws IncompatibilityException {
		insertBlockConstraint(constraint, get(targetBlock).declaredConstraints, false);
	}
	
	private void insertBlockConstraint(BlockConstraint constraint, List<BlockConstraint> dest, boolean overriding) throws IncompatibilityException {
		for (int index = 0; index < dest.size(); index++) {
			BlockConstraint a = BlockConstraint.combine(dest.get(index), constraint, overriding);
			
			if (a != null) {
				dest.set(index, a);
				return;
			}
		}
		
		dest.add(constraint.createCopy());
	}
	
	/**
	 * Recursively computes all classes that are referenced indirectly
	 * by the classes that have been added so far.
	 * The result includes classes that are nested inside IBD2Targets (variants). 
	 */
	private Set<Class<?>> getIndirectlyReferencedClasses() {
		Set<Class<?>> beenHere = new HashSet<Class<?>>();
		beenHere.addAll(blockPerUserDefinedClz.keySet());
		
		Set<Class<?>> fringe = new HashSet<Class<?>>();
		Set<Class<?>> newFringe = new HashSet<Class<?>>();
		fringe.addAll(blockPerUserDefinedClz.keySet());
		
		while (fringe.size() > 0) {
			newFringe.clear();
			
			for (Class<?> f : fringe) {
				for (Class<?> rt : SourceType.getReferencedTypes(f)) {
					if (beenHere.add(rt)) {
						newFringe.add(rt);
					}
				}
			}
			
			fringe.clear();
			fringe.addAll(newFringe);
		}
		
		beenHere.removeAll(blockPerUserDefinedClz.keySet());
		return beenHere;
	}
	
	public void process() throws ReflectionException {
		//Add indirectly referenced classes:
		for (Class<?> irc : getIndirectlyReferencedClasses()) {
			if (irc.getDeclaringClass() != null) {
				if (IBD2Target.class.isAssignableFrom(irc.getDeclaringClass().getSuperclass())) {
					BlockId blockId = new BlockId(irc);
					Block block = new Block(this, blockId);
					blockPerId.put(blockId, block);
					blockPerUserDefinedClz.put(irc, block);
					BlockCache blockCache = new BlockCache(block);
					blockCache.userDefinedClzs.add(irc);
					cachePerBlock.put(block, blockCache);
				}
			} else {
				add(irc);
			}
		}
		
		//Clean up data from a previous execution, just in case:
		for (BlockCache cache : cachePerBlock.values()) {
			cache.clearProcessed();
		}
		
		// Generate aspects + connection points from the user-defined classes:
		for (Map.Entry<Block, BlockCache> entry : cachePerBlock.entrySet()) {
			for (Class<?> clz : entry.getValue().userDefinedClzs) {
				try {
					AggregationConstraint.fromDiagram(this, clz);
					ASALPortConstraint.fromDiagram(this, clz);
					AssociationConstraint.fromDiagram(this, clz);
					GeneralizationConstraint.fromDiagram(this, clz);
				} catch (IncompatibilityException e) {
					throw new Error(e);
				}
				
				switch (SourceType.get(clz)) {
					case BDD:
						//We want to check the validity of bidirectional associations as soon as possible:
						BDD bdd = BDD.class.cast(ClassUtils.createStdInstance(clz.getDeclaringClass()));
						bdd.initBidirMap(this, clz.getDeclaringClass());
						bdd.initBidirs();
						//TODO: check that all associations are bidirectional!!
						break;
					case IBD1:
						break;
					case IBD2_INTERFACE:
						break;
					case IBD2:
						try {
							System.out.println("Processing " + clz.getCanonicalName() + "...");
							entry.getValue().contextPts.addAll(ConnectionPts.getConnectionPts(entry.getKey(), clz));
						} catch (ConnectionPtException e) {
							throw new Error(e);
						}
						break;
				}
			}
		}
		
		// Add inherited constraints + connection points to each block:
		for (Map.Entry<Block, BlockCache> entry : cachePerBlock.entrySet()) {
			for (Block b : entry.getKey().getLineage()) {
				for (BlockConstraint a : get(b).declaredConstraints) {
					if (!(a instanceof GeneralizationConstraint) || b == entry.getKey()) {
						try {
							insertBlockConstraint(a, entry.getValue().inheritedConstraints, true);
						} catch (IncompatibilityException e) {
							throw new Error(e);
						}
					}
				}
				
				if (b != entry.getKey()) {
					entry.getValue().contextPts.addAll(get(b).contextPts);
				}
			}
		}
		
		// For convenience, we want to know which connection points a block owns:
		for (Map.Entry<Block, BlockCache> entry : cachePerBlock.entrySet()) {
			for (ConnectionPt pt : entry.getValue().contextPts) {
				get(pt.ownerBlock).ownedPts.add(pt);
			}
		}
		
		// Confirm that blocks are used consistently as either parts or contexts.
		// We check the entire lineage of a block:
		for (Block block : blockPerUserDefinedClz.values()) {
			Map<SourceType, Set<Class<?>>> clzsPerSource = SourceType.categorizeUserDefinedClzs(block);
			clzsPerSource.remove(SourceType.BDD);
			
			if (clzsPerSource.isEmpty()) {
				throw new Error("Block \"" + block.id + "\" should be an IBD1, IBD2, or IBD2 interface!");
			}
			
			if (clzsPerSource.size() > 1) {
				String msg = "Block \"" + block.id + "\" can only be an IBD1, an IBD2, or an IBD2 interface:";
				
				for (Map.Entry<SourceType, Set<Class<?>>> e : clzsPerSource.entrySet()) {
					for (Class<?> c : e.getValue()) {
						msg += "\n\t" + e.getKey().prettyName + " " + c.getCanonicalName();
					}
				}
				
				throw new Error(msg);
			}
			
			block.getCache().sourceType = clzsPerSource.entrySet().iterator().next().getKey();
		}
		
		// Confirm that flow specs are consistent:
		FlowSpecExprMap flowSpecMap = new FlowSpecExprMap();
		
		for (Block block : blockPerUserDefinedClz.values()) {
			flowSpecMap.add(block);
		}
		
		flowSpecMap.process();
		
		// Store information about flow specs per connection point in the cache:
		flowSpecsPerOwnedPt.clear();
		
		for (ConnectionPt.InterfacePt pt : flowSpecMap.getConnectionPts()) {
			flowSpecsPerOwnedPt.put(pt, flowSpecMap.getReprFlowSpecs(pt));
		}
		
		// Create flow specifications themselves:
		flowSpecPerName.clear();
		
		for (String reprInterfaceName : flowSpecMap.getReprFlowSpecNames()) {
			flowSpecPerName.put(reprInterfaceName, new Interface(reprInterfaceName));
		}
		
		for (ConnectionPt.PrimitivePt pt : flowSpecMap.getPrimPts()) {
			for (ConnectionPt connectedPt : pt.getConnectedPts()) {
				if (connectedPt instanceof ConnectionPt.InterfacePt) {
					for (FlowSpecExpr expr : flowSpecMap.getReprFlowSpecs((ConnectionPt.InterfacePt)connectedPt)) {
						Interface itf = flowSpecPerName.get(expr.flowSpecName);
						
						if (pt instanceof ConnectionPt.InPortPt) {
							ConnectionPt.InPortPt p = (ConnectionPt.InPortPt) pt;
							Field f = new Field(pt, Port.getSuffix(pt.field.getName()), p.getType(), expr.isConjugate ? Dir.OUT : Dir.IN);
							itf.addField(f);
							continue;
						}
						
						if (pt instanceof ConnectionPt.OutPortPt) {
							ConnectionPt.OutPortPt p = (ConnectionPt.OutPortPt) pt;
							Field f = new Field(pt, Port.getSuffix(pt.field.getName()), p.getType(), expr.isConjugate ? Dir.IN : Dir.OUT);
							itf.addField(f);
							continue;
						}
						
						throw new Error("Should not happen!");
					}
				}
			}
		}
		
		for (Map.Entry<String, Interface> expr : flowSpecPerName.entrySet()) {
			expr.getValue().print();
		}
		
		//Determine interactions per primitive flow port:
		InteractionMap interactionMap = new InteractionMap();
		
		for (Block block : blockPerUserDefinedClz.values()) {
			interactionMap.add(block);
		}
		
		interactionMap.process();
		
		interfaceNamesPerPrimPt.clear();
		
		for (ConnectionPt.PrimitivePt pt : interactionMap.getPrimPts()) {
			interfaceNamesPerPrimPt.put(pt, interactionMap.getInteraction(pt).getInteractionNames());
		}
		
		// Create partial models and check them for conformance:
//		for (Block block : new HashSet<Block>(blockPerUserDefinedClz.values())) {
//			System.out.println("Checking " + block.id + " for conformance...");
//			Model m = new Model(this);
//			m.add("x", block, RenamingScheme.SINGLETONS);
//			m.finish();
//			m.checkConformance();
//			
//			new GraphVizModelPrinter(m).add("debug/conformance.block." + block.id + ".gv").printAndPop();
//		}
	}
}
