package lib.blocks.models;

import java.util.*;

import lib.blocks.common.*;
import lib.blocks.constraints.*;
import lib.blocks.models.IBDInstances.*;

public class Model extends ComposableModel {
	public static class IBD2ConfProb {
		public final IBD2Port port;
		public final AssociationConstraint constraint;
		public final ViolatingIBD2Exception exception;
		
		public IBD2ConfProb(IBD2Port port, AssociationConstraint constraint, ViolatingIBD2Exception exception) {
			this.port = port;
			this.constraint = constraint;
			this.exception = exception;
		}
	}
	
	private Map<IBD2Port, Set<IBD2ConfProb>> confProbsPerIBD2Port;
	
	public Model(ModelLib lib) {
		super(lib);
		
		confProbsPerIBD2Port = new HashMap<IBDInstances.IBD2Port, Set<IBD2ConfProb>>();
	}
	
	private static void confirmConjugation(IBD1Port p1, IBD1Port p2, List<IBD2Flow> seq) {
		if (p1.type != p2.type) {
			throw new Error("Port and its conjugate must have matching types!");
		}
		
		if (p1.dir != p2.dir.getOpposite()) {
			String msg = "Ports and their conjugates must have opposite directions, but found";
			msg += "\n\t\t" + p1.dir.text + " " + p1.owner.getType().id + "." + p1.name;
			
			for (Class<?> userDefinedClz : p1.owner.getType().getUserDefinedClzs()) {
				msg += "\n\t\t\tfrom \"" + userDefinedClz.getCanonicalName() + "\"";
			}
			
			msg += "\n\t\t" + p2.dir.text + " " + p2.owner.getType().id + "." + p2.name;
			
			for (Class<?> userDefinedClz : p2.owner.getType().getUserDefinedClzs()) {
				msg += "\n\t\t\tfrom \"" + userDefinedClz.getCanonicalName() + "\"";
			}
			
			if (seq != null) {
				msg += "\nconnected via";
				
				for (IBD2Flow s : seq) {
					msg += "\n\t\t" + s.v1.owner.getType().id + "." + s.v1.id + " -> " + s.v2.owner.getType().id + "." + s.v2.id;
					
					for (ConnectionPt.InterfacePt ip : s.v1.getLegacyPts()) {
						msg += "\n\t\t\tfirst found in \"" + ip.diagramClz.getCanonicalName() + "\"";
					}
					
					for (ConnectionPt.InterfacePt ip : s.v2.getLegacyPts()) {
						msg += "\n\t\t\tsecond found in \"" + ip.diagramClz.getCanonicalName() + "\"";
					}
				}
			}
			
			throw new Error(msg);
		}
	}
	
	public Set<IBD1Port> getConjugateIBD1Ports(IBD1Port port) {
		Set<IBD1Port> result = new HashSet<IBD1Port>();
		String suffix = Port.getSuffix(port.name);
		
		for (IBD1Flow f : getIBD1Flows()) {
			if (f.v1 == port) {
				confirmConjugation(f.v2, port, null);
				result.add(f.v2);
			} else {
				if (f.v2 == port) {
					confirmConjugation(f.v1, port, null);
					result.add(f.v1);
				}
			}
		}
		
		for (IBD1ToIBD2Flow f1 : getIBD1ToIBD2Flows()) {
			if (f1.v1 == port) {
				List<IBD2Flow> seq = getIBD2FlowSeq(f1.v2);
				
				if (seq.size() > 0) {
					IBD2Port otherEnd = seq.get(seq.size() - 1).v2;
					
					for (IBD1ToIBD2Flow f2 : getIBD1ToIBD2Flows()) {
						if (f2.v2 == otherEnd) {
							if (Port.getSuffix(f2.v1.name).equals(suffix)) {
								confirmConjugation(f2.v1, port, seq);
								result.add(f2.v1);
							}
						}
					}
				}
			}
		}
		
		return result;
	}
	
	public IBD2Port getConjugateIBD2Port(IBD2Port port) {
		List<IBD2Flow> seq = getIBD2FlowSeq(port);
		return seq.size() > 0 ? seq.get(seq.size() - 1).v2 : null;
	}
	
	public List<IBD2Flow> getIBD2FlowSeq(IBD2Port initialPort) {
		for (IBD2Flow f : getIBD2Flows()) {
			if (f.v1 == initialPort) {
				return getIBD2FlowSeq(f);
			}
		}
		
		return Collections.emptyList();
	}
	
	public List<IBD2Flow> getIBD2FlowSeq(IBD2Flow initialFlow) {
		List<IBD2Flow> result = new ArrayList<IBD2Flow>();
		result.add(initialFlow);
		boolean done = false;
		
		while (!done) {
			done = true;
			
			IBD2Flow firstFlow = result.get(0);
			IBD2Flow lastFlow = result.get(result.size() - 1);
			
			for (IBD2Flow f : getIBD2Flows()) {
				if (result.size() > 1 && f.v1 == lastFlow.v2 && f.v2 == firstFlow.v1) {
					throw new Error("Interface flow port must not be part of a flow cycle!");
				}
				
				if (f.v2 == firstFlow.v1 && f.v1 != firstFlow.v2) {
					if (result.get(0) != firstFlow) {
						throw new Error("Interface flow port is connected to too many other interface flow ports!");
					}
					
					result.add(0, f);
					done = false;
				}
				
				if (f.v1 == lastFlow.v2 && f.v2 != lastFlow.v1) {
					if (result.get(result.size() - 1) != lastFlow) {
						throw new Error("Interface flow port is connected to too many other interface flow ports!");
					}
					
					result.add(f);
					done = false;
				}
			}
		}
		
		return result;
	}
	
	public void initIBD2ConfProbs() {
		confProbsPerIBD2Port.clear();
		List<Exception> problems = new ArrayList<Exception>();
		addIBD2ConformanceProblems(problems);
	}
	
	public void checkConformance() {
		List<Exception> problems = new ArrayList<Exception>();
		addIBD1ConformanceProblems(problems);
		addIBD2ConformanceProblems(problems);
		
		if (problems.size() > 0) {
			for (Exception problem : problems) {
				problem.printStackTrace();
			}
			
			throw new Error("Model has problems!");
		}
	}
	
	private void addIBD1ConformanceProblems(List<Exception> problems) {
		Map<Block, Set<IBD1Instance>> ibd1sPerType = new HashMap<Block, Set<IBD1Instance>>();
		
		for (IBD1Instance ibd1 : getIBD1Instances()) {
			Set<IBD1Instance> names = ibd1sPerType.get(ibd1.getType());
			
			if (names == null) {
				names = new HashSet<IBD1Instance>();
				ibd1sPerType.put(ibd1.getType(), names);
			}
			
			names.add(ibd1);
		}
		
		for (Map.Entry<Block, Set<IBD1Instance>> entry : ibd1sPerType.entrySet()) {
			for (BlockConstraint constraint : entry.getKey().getInheritedConstraints()) {
				for (IBD1Instance ibd1 : entry.getValue()) {
					try {
						constraint.checkConformance(this, ibd1);
					} catch (ViolatedConstraintException e) {
						problems.add(new ViolatingIBD1Exception(ibd1, e));
					}
				}
			}
		}
	}
	
	private void addIBD2ConformanceProblems(List<Exception> problems) {
		Map<Block, Set<IBD2Instance>> ibd2sPerType = new HashMap<Block, Set<IBD2Instance>>();
		
		for (IBD2Instance ibd2 : getIBD2Instances()) {
			Set<IBD2Instance> names = ibd2sPerType.get(ibd2.getType());
			
			if (names == null) {
				names = new HashSet<IBD2Instance>();
				ibd2sPerType.put(ibd2.getType(), names);
			}
			
			names.add(ibd2);
		}
		
		for (Map.Entry<Block, Set<IBD2Instance>> entry : ibd2sPerType.entrySet()) {
			for (BlockConstraint constraint : entry.getKey().getInheritedConstraints()) {
				for (IBD2Instance ibd2 : entry.getValue()) {
					try {
						constraint.checkConformance(this, ibd2);
					} catch (ViolatedConstraintException e) {
						ViolatingIBD2Exception ve = new ViolatingIBD2Exception(ibd2, e);
						
						if (constraint instanceof AssociationConstraint) {
							AssociationConstraint ac = (AssociationConstraint)constraint;
							
							for (IBD2Port port : ibd2.getPortPerId().values()) {
								if (port.id.name.equals(ac.getName())) {
									addIBD2ConfProb(new IBD2ConfProb(port, ac, ve));
								}
							}
						}
						
						problems.add(ve);
					}
				}
			}
		}
	}
	
	private void addIBD2ConfProb(IBD2ConfProb prob) {
		Set<IBD2ConfProb> probs = confProbsPerIBD2Port.get(prob.port);
		
		if (probs == null) {
			probs = new HashSet<IBD2ConfProb>();
			confProbsPerIBD2Port.put(prob.port, probs);
		}
		
		for (IBD2ConfProb p : probs) {
			if (p.constraint == prob.constraint) {
				return;
			}
		}
		
		probs.add(prob);
	}
	
	public Set<IBD2ConfProb> getIBD2ConfProbs(IBD2Port port) {
		Set<IBD2ConfProb> result = confProbsPerIBD2Port.get(port);
		
		if (result != null) {
			return Collections.unmodifiableSet(result);
		}
		
		return Collections.emptySet();
	}
}


