package lib.blocks.models;

import java.util.*;

import lib.asal.ASALDataType;
import lib.asal.parsing.ASALException;
import lib.behave.StateMachine;
import lib.behave.proto.*;
import lib.blocks.common.ConnectionPt;
import lib.blocks.common.Port;
import lib.blocks.constraints.AssociationConstraint;
import lib.blocks.constraints.BlockConstraint;
import lib.utils.*;

/**
 * Manages all IBD elements that are instantiated by the user (via {@link ComposableModel}).
 */
public class IBDInstances {
	public static class IBD1Instance {
		private Map<String, IBD1Port> portPerName;
		private Block type;
		private TritoStateMachine stateMachine;
		private boolean isInstantiated;
		
		public final NameIndexId id;
		
		private IBD1Instance(NameIndexId id, Block type, boolean isInstantiated) {
			this.id = id;
			this.isInstantiated = isInstantiated;
			this.type = type;
			
			portPerName = new HashMap<String, IBD1Port>();
		}
		
		public TritoStateMachine getStateMachine() {
			return stateMachine;
		}
		
		public void setStateMachine(Class<? extends StateMachine> stateMachineClz) throws ASALException {
			if (this.stateMachine != null) {
				throw new Error("State machine has already been set!");
			}
			
			ProtoStateMachine proto = new ProtoStateMachine(stateMachineClz, type);
			DeuteroStateMachine deutero = new DeuteroStateMachine(proto);
			TritoStateMachine trito = new TritoStateMachine(deutero);
			
			stateMachine = trito;
		}
		
		public boolean isInstantiated() {
			return isInstantiated;
		}
		
		public Block getType() {
			return type;
		}
		
		public Map<String, IBD1Port> getPortPerName() {
			return Collections.unmodifiableMap(portPerName);
		}
		
		private IBD1Port getOrCreatePort(ConnectionPt.PrimitivePt pp) {
			IBD1Port result = portPerName.get(pp.field.getName());
			
			if (result == null) {
				result = new IBD1Port(this, pp);
				portPerName.put(result.name, result);
			}
			
			result.legacyPts.add(pp);
			return result;
		}
	}
	
	public static class IBD1Port {
		private Set<ConnectionPt.PrimitivePt> legacyPts;
		private Set<IBD2Port> attachedPorts;
		
		public final IBD1Instance owner;
		public final ConnectionPt.PrimitivePt someLegacyPt;
		public final String name;
		public final ASALDataType type;
		public final Dir dir;
		
		private IBD1Port(IBD1Instance owner, ConnectionPt.PrimitivePt pp) {
			this.owner = owner;
			
			someLegacyPt = pp;
			name = pp.field.getName();
			type = pp.getType();
			dir = pp.getDir();
			
			attachedPorts = new HashSet<IBD2Port>();
			legacyPts = new HashSet<ConnectionPt.PrimitivePt>();
			legacyPts.add(pp);
		}
		
		public Set<ConnectionPt.PrimitivePt> getLegacyPts() {
			return Collections.unmodifiableSet(legacyPts);
		}
		
		public Set<IBD2Port> getAttachedPorts() {
			return Collections.unmodifiableSet(attachedPorts);
		}
		
		@Override
		public String toString() {
			String result = "" + name + "~" + hashCode();
			
			for (ConnectionPt.PrimitivePt ip : legacyPts) {
				result += " (from " + ip.diagramClz.getCanonicalName() + ")";
			}
			
			return result;
		}
	}
	
	public static class IBD1Flow {
		public final IBD1Port v1;
		public final IBD1Port v2;
		
		private IBD1Flow(IBD1Port v1, IBD1Port v2) {
			this.v1 = v1;
			this.v2 = v2;
		}
		
		@Override
		public int hashCode() {
			return Objects.hash(v1, v2);
		}
		
		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (!(obj instanceof IBD1Flow)) {
				return false;
			}
			IBD1Flow other = (IBD1Flow) obj;
			return Objects.equals(v1, other.v1) && Objects.equals(v2, other.v2);
		}
	}
	
	public static class IBD1ToIBD2Flow {
		public final IBD1Port v1;
		public final IBD2Port v2;
		
		private IBD1ToIBD2Flow(IBD1Port v1, IBD2Port v2) {
			this.v1 = v1;
			this.v2 = v2;
		}
		
		@Override
		public int hashCode() {
			return Objects.hash(v1, v2);
		}
		
		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (!(obj instanceof IBD1ToIBD2Flow)) {
				return false;
			}
			IBD1ToIBD2Flow other = (IBD1ToIBD2Flow) obj;
			return Objects.equals(v1, other.v1) && Objects.equals(v2, other.v2);
		}
	}
	
	public static class IBD2Instance {
		private Map<NameIndexId, IBD2Port> portPerId;
		private Set<IBD1Instance> childIBD1s;
		private Set<IBD2Instance> childIBD2s;
		private Block type;
		private boolean isInstantiated;
		
		public final NameIndexId id;
		
		private IBD2Instance(NameIndexId id, Block type, boolean isInstantiated) {
			this.id = id;
			this.isInstantiated = isInstantiated;
			this.type = type;
			
			portPerId = new HashMap<NameIndexId, IBD2Port>();
			childIBD1s = new HashSet<IBD1Instance>();
			childIBD2s = new HashSet<IBD2Instance>();
		}
		
		public Block getType() {
			return type;
		}
		
		public boolean isInstantiated() {
			return isInstantiated;
		}
		
		public Map<NameIndexId, IBD2Port> getPortPerId() {
			return Collections.unmodifiableMap(portPerId);
		}
		
		public Set<IBD1Instance> getChildIBD1s() {
			return Collections.unmodifiableSet(childIBD1s);
		}
		
		public Set<IBD2Instance> getChildIBD2s() {
			return Collections.unmodifiableSet(childIBD2s);
		}
		
		private IBD2Port getOrCreatePort(ConnectionPt.InterfacePt ip) {
			if (ip == null) {
				System.out.println("ip == null");
			}
			
			if (ip.getPortId() == null) {
				System.out.println("ip.getPortId() == null");
			}
			
			if (portPerId == null) {
				System.out.println("portPerId == null");
			}
			
			IBD2Port result = portPerId.get(ip.getPortId());
			
			if (result == null) {
				result = new IBD2Port(this, ip);
				portPerId.put(ip.getPortId(), result);
			}
			
			result.legacyPts.add(ip);
			return result;
		}
	}
	
	public static class IBD2Port {
		private Set<ConnectionPt.InterfacePt> legacyPts;
		private Set<IBD2Req> reqs;
		
		public final IBD2Instance owner;
		public final ConnectionPt.InterfacePt someLegacyPt;
		public final NameIndexId id;
		
		private IBD2Port(IBD2Instance owner, ConnectionPt.InterfacePt ip) {
			this.owner = owner;
			
			someLegacyPt = ip;
			id = ip.getPortId();
			
			legacyPts = new HashSet<ConnectionPt.InterfacePt>();
			legacyPts.add(ip);
			
			reqs = new HashSet<IBD2Req>();
		}
		
		public Set<ConnectionPt.InterfacePt> getLegacyPts() {
			return Collections.unmodifiableSet(legacyPts);
		}
		
		private void addReq(AssociationConstraint c) {
			for (IBD2Req req : reqs) {
				if (req.constraint == c) {
					return;
				}
			}
			
			reqs.add(new IBD2Req(this, c));
		}
		
		@Override
		public String toString() {
			String result = "" + id + "~" + hashCode();
			
			for (ConnectionPt.InterfacePt ip : legacyPts) {
				result += " (from " + ip.diagramClz.getCanonicalName() + ")";
			}
			
			return result;
		}
	}
	
	public static class IBD2Flow {
		public final IBD2Port v1;
		public final Dir dir1;
		public final IBD2Port v2;
		public final Dir dir2;
		
		public IBD2Flow(IBD2Port v1, Dir dir1, IBD2Port v2, Dir dir2) {
			this.v1 = v1;
			this.dir1 = dir1;
			this.v2 = v2;
			this.dir2 = dir2;
		}
		
		@Override
		public int hashCode() {
			return Objects.hash(dir1, dir2, v1, v2);
		}
		
		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (!(obj instanceof IBD2Flow)) {
				return false;
			}
			IBD2Flow other = (IBD2Flow) obj;
			return dir1 == other.dir1 && dir2 == other.dir2 && Objects.equals(v1, other.v1) && Objects.equals(v2, other.v2);
		}
	}
	
	public static class IBD2Req {
		public final IBD2Port port;
		public final AssociationConstraint constraint;
		
		public IBD2Req(IBD2Port port, AssociationConstraint constraint) {
			this.port = port;
			this.constraint = constraint;
		}
	}
	
	private Map<NameIndexId, IBD1Instance> ibd1InstancePerId;
	private Map<NameIndexId, IBD2Instance> ibd2InstancePerId;
	private Set<IBD1Flow> ibd1Flows;
	private Set<IBD2Flow> ibd2Flows;
	private Set<IBD1ToIBD2Flow> ibd1ToIBD2Flows;
	private Set<IBD2Req> ibd2Reqs;
	
	public final ModelLib lib;
	
	public IBDInstances(ModelLib lib) {
		this.lib = lib;
		
		ibd1InstancePerId = new HashMap<NameIndexId, IBD1Instance>();
		ibd2InstancePerId = new HashMap<NameIndexId, IBD2Instance>();
		ibd1Flows = new HashSet<IBD1Flow>();
		ibd2Flows = new HashSet<IBD2Flow>();
		ibd1ToIBD2Flows = new HashSet<IBD1ToIBD2Flow>();
		ibd2Reqs = new HashSet<IBD2Req>();
	}
	
	public void clear() {
		ibd1InstancePerId.clear();
		ibd2InstancePerId.clear();
		ibd1Flows.clear();
		ibd2Flows.clear();
		ibd1ToIBD2Flows.clear();
		ibd2Reqs.clear();
	}
	
	public Collection<IBD1Instance> getIBD1Instances() {
		return Collections.unmodifiableCollection(ibd1InstancePerId.values());
	}
	
	public Collection<IBD2Instance> getIBD2Instances() {
		return Collections.unmodifiableCollection(ibd2InstancePerId.values());
	}
	
	public Set<IBD1Flow> getIBD1Flows() {
		return Collections.unmodifiableSet(ibd1Flows);
	}
	
	public Set<IBD2Flow> getIBD2Flows() {
		return Collections.unmodifiableSet(ibd2Flows);
	}
	
	public Set<IBD1ToIBD2Flow> getIBD1ToIBD2Flows() {
		return Collections.unmodifiableSet(ibd1ToIBD2Flows);
	}
	
	public void addIBDInstance(NameIndexId instanceName, Block instanceType) {
		addIBDInstance(instanceName, instanceType, "\t");
	}
	
	private void addIBDInstance(NameIndexId instanceId, Block instanceType, String indent) {
		System.out.println(indent + "Instantiating " + instanceId + " of type " + instanceType.id + "!");
		
		switch (instanceType.getSourceType()) {
			case BDD:
				throw new Error("Should not happen!");
			case IBD2_INTERFACE:
				throw new Error("Should not happen!");
			case IBD1:
				IBD1Instance ibd1 = getOrCreateIBD1(instanceId, instanceType, true);
				
				for (Block superType : instanceType.getLineage()) {
					for (ConnectionPt pt : superType.getOwnedPts()) {
						if (pt instanceof ConnectionPt.PrimitivePt) {
							ConnectionPt.PrimitivePt pp = (ConnectionPt.PrimitivePt) pt;
							IBD1Port p = ibd1.getOrCreatePort(pp);
							p.legacyPts.add(pp);
							continue;
						}
						
						throw new Error("Should not happen!");
					}
				}
				
				break;
			case IBD2:
				IBD2Instance ibd2 = getOrCreateIBD2(instanceId, instanceType, true, indent + "\t1 ");
				
				Map<ConnectionPt.PrimitivePt, IBD1Port> portPerPrimitivePt = new HashMap<ConnectionPt.PrimitivePt, IBD1Port>();
				Map<ConnectionPt.InterfacePt, IBD2Port> portPerInterfacePt = new HashMap<ConnectionPt.InterfacePt, IBD2Port>();
				
				for (ConnectionPt pt : instanceType.getContextPts()) {
					if (pt instanceof ConnectionPt.PrimitivePt) {
						ConnectionPt.PrimitivePt pp = (ConnectionPt.PrimitivePt) pt;
						IBD1Instance childIBD1 = getOrCreateIBD1(pp.getMemberId(), pp.ownerBlock, false);
						ibd2.childIBD1s.add(childIBD1);
						
						IBD1Port p = childIBD1.getOrCreatePort(pp);
						portPerPrimitivePt.put(pp, p);
						continue;
					}
					
					if (pt instanceof ConnectionPt.ExposingPt) {
						ConnectionPt.ExposingPt ep = (ConnectionPt.ExposingPt) pt;
						IBD2Port p = ibd2.getOrCreatePort(ep);
						portPerInterfacePt.put(ep, p);
						continue;
					}
					
					if (pt instanceof ConnectionPt.ExposedPt) {
						ConnectionPt.ExposedPt ep = (ConnectionPt.ExposedPt) pt;
						IBD2Instance childIBD2 = getOrCreateIBD2(ep.getMemberId(), ep.ownerBlock, false, indent + "\t3 ");
						ibd2.childIBD2s.add(childIBD2);
						
						IBD2Port p = childIBD2.getOrCreatePort(ep);
						portPerInterfacePt.put(ep, p);
						continue;
					}
				}
				
				// Create flows:
				for (ConnectionPt pt1 : instanceType.getContextPts()) {
					if (pt1 instanceof ConnectionPt.PrimitivePt) {
						IBD1Port p1 = portPerPrimitivePt.get((ConnectionPt.PrimitivePt) pt1);
						
						for (ConnectionPt pt2 : pt1.getConnectedPts()) {
							//We only do output ports so that flows only exist in one direction:
							if (pt2 instanceof ConnectionPt.OutPortPt) {
								IBD1Port p2 = portPerPrimitivePt.get((ConnectionPt.PrimitivePt) pt2);
								addIBD1Flow(p1, p2);
								continue;
							}
							
							if (pt2 instanceof ConnectionPt.ExposingPt) {
								IBD2Port p2 = portPerInterfacePt.get((ConnectionPt.ExposingPt) pt2);
								addIBD1ToIBD2Flow(p1, p2);
								continue;
							}
						}
						
						continue;
					}
					
					if (pt1 instanceof ConnectionPt.ExposedPt) {
						IBD2Port p1 = portPerInterfacePt.get((ConnectionPt.ExposedPt) pt1);
						
						for (ConnectionPt pt2 : pt1.getConnectedPts()) {
							if (pt2 instanceof ConnectionPt.ExposingPt) {
								throw new Error("No supported!");
//								IBD2Port p2 = portPerInterfacePt.get((ConnectionPt.ExposingPt) pt2);
//								Dir dir1 = pt1 instanceof ConnectionPt.ExposedPt ? Dir.OUT : Dir.IN;
//								Dir dir2 = pt2 instanceof ConnectionPt.ExposedPt ? Dir.OUT : Dir.IN;
//								ibd2Flows.add(new IBD2Flow(p1, dir1, p2, dir2));
//								continue;
							}
							
							if (pt2 instanceof ConnectionPt.ExposedPt) {
								IBD2Port p2 = portPerInterfacePt.get((ConnectionPt.ExposedPt) pt2);
								addIBD2Flow(p1, p2);
								continue;
							}
						}
						
						continue;
					}
				}
				
				// Create requirements (which can be removed later if they are satisfied):
				for (BlockConstraint c : instanceType.getInheritedConstraints()) {
					if (c instanceof AssociationConstraint) {
						AssociationConstraint ac = (AssociationConstraint)c;
						
						for (Map.Entry<ConnectionPt.InterfacePt, IBD2Port> entry : portPerInterfacePt.entrySet()) {
							if (entry.getKey().field.getName().equals(ac.getName())) {
								entry.getValue().addReq(ac);
							}
						}
					}
				}
				
				// Recursively add components (TODO: only COMPOSITIONAL aggregations!!):
				for (IBD1Instance childIBD1 : ibd2.childIBD1s) {
					addIBDInstance(childIBD1.id, childIBD1.type, indent + "\t");
				}
				
				for (IBD2Instance childIBD2 : ibd2.childIBD2s) {
					addIBDInstance(childIBD2.id, childIBD2.type, indent + "\t");
				}
				
				break;
		}
	}
	
	private IBD1Flow getIBD1Flow(IBD1Port v1, IBD1Port v2) {
		for (IBD1Flow f : ibd1Flows) {
			if (f.v1 == v1 && f.v2 == v2) {
				return f;
			}
		}
		
		return null;
	}
	
	private IBD2Flow getIBD2Flow(IBD2Port v1, IBD2Port v2) {
		for (IBD2Flow f : ibd2Flows) {
			if (f.v1 == v1 && f.v2 == v2) {
				return f;
			}
			
			if (f.v2 == v1 && f.v1 == v2) {
				return f;
			}
		}
		
		return null;
	}
	
	private IBD1ToIBD2Flow getIBD1ToIBD2Flow(IBD1Port v1, IBD2Port v2) {
		for (IBD1ToIBD2Flow f : ibd1ToIBD2Flows) {
			if (f.v1 == v1 && f.v2 == v2) {
				return f;
			}
		}
		
		return null;
	}
	
	private void addIBD1Flow(IBD1Port v1, IBD1Port v2) {
		if (v1.someLegacyPt.getDir() == Dir.OUT && v2.someLegacyPt.getDir() == Dir.IN) {
			if (getIBD1Flow(v1, v2) == null) {
				ibd1Flows.add(new IBD1Flow(v1, v2));
			}
			
			return;
		}
		
		if (v1.someLegacyPt.getDir() == Dir.IN && v2.someLegacyPt.getDir() == Dir.OUT) {
			addIBD1Flow(v2, v1);
			return;
		}
		
		String msg = "Should not connect ports:";
		msg += "\n\t" + v1.toString();
		msg += "\n\t" + v2.toString();
		
		throw new Error(msg);
	}
	
	private void addIBD1ToIBD2Flow(IBD1Port v1, IBD2Port v2) {
		if (getIBD1ToIBD2Flow(v1, v2) == null) {
			IBD1ToIBD2Flow f12 = new IBD1ToIBD2Flow(v1, v2);
			ibd1ToIBD2Flows.add(f12);
			
			v1.attachedPorts.add(v2);
			
			for (IBD2Flow f22 : ibd2Flows) {
				for (IBD1ToIBD2Flow f21 : ibd1ToIBD2Flows) {
					if ((f12.v2 == f22.v1 && f21.v2 == f22.v2) || (f12.v2 == f22.v2 && f21.v2 == f22.v1)) {
						if (Port.sameSuffix(f12.v1.name, f21.v1.name)) {
							addIBD1Flow(f12.v1, f21.v1);
						}
					}
				}
			}
		}
	}
	
	private void addIBD2Flow(IBD2Port v1, IBD2Port v2) {
		if (getIBD2Flow(v1, v2) == null) {
			ibd2Flows.add(new IBD2Flow(v1, Dir.OUT, v2, Dir.OUT));
			
			for (IBD1ToIBD2Flow f12 : ibd1ToIBD2Flows) {
				for (IBD1ToIBD2Flow f21 : ibd1ToIBD2Flows) {
					if ((f21.v2 == v2 && f12.v2 == v1) || (f12.v2 == v2 && f21.v2 == v1)) {
						if (Port.sameSuffix(f21.v1.name, f12.v1.name)) {
							addIBD1Flow(f21.v1, f12.v1);
						}
					}
				}
			}
		}
	}
	
	private IBD1Instance getOrCreateIBD1(NameIndexId id, Block type, boolean isInstantiated) {
		IBD1Instance result = ibd1InstancePerId.get(id);
		
		if (result != null) {
			result.isInstantiated = result.isInstantiated || isInstantiated;
			result.type = Block.getNarrowBlock(result.type, type, id + "\n" + result.id + "\n");
		} else {
			result = new IBD1Instance(id, type, isInstantiated);
			ibd1InstancePerId.put(id, result);
		}
		
		return result;
	}
	
	private IBD2Instance getOrCreateIBD2(NameIndexId id, Block type, boolean isInstantiated, String indent) {
		IBD2Instance result = ibd2InstancePerId.get(id);
		
		if (result != null) {
			System.out.println(indent + "Reinstantiating " + id + " of type " + type.id + "!");
			result.isInstantiated = result.isInstantiated || isInstantiated;
			result.type = Block.getNarrowBlock(result.type, type, id + "\n" + result.id + "\n");
		} else {
			System.out.println(indent + "Registering new instance " + id + " of type " + type.id + "!");
			result = new IBD2Instance(id, type, isInstantiated);
			ibd2InstancePerId.put(id, result);
		}
		
		return result;
	}
	
	public void print() {
		for (IBD2Instance ibd2 : ibd2InstancePerId.values()) {
			System.out.println("IBD2 " + ibd2.id + ": " + ibd2.type.id);
			
			for (Class<?> clz : ibd2.type.getUserDefinedClzs()) {
				System.out.println("\tfrom " + clz.getCanonicalName());
			}
			
			for (Map.Entry<NameIndexId, IBD2Port> e : ibd2.portPerId.entrySet()) {
				System.out.println("\towns " + e.getValue());
			}
			
			for (IBD2Flow f2 : ibd2Flows) {
				if (f2.v1.owner == ibd2) {
					System.out.println("\t" + f2.v1 + "\n\t\t<--> " + f2.v2);
				}
				
				if (f2.v2.owner == ibd2) {
					System.out.println("\t" + f2.v2 + "\n\t\t<--> " + f2.v1);
				}
			}
		}
		
		for (IBD1ToIBD2Flow f : ibd1ToIBD2Flows) {
			System.out.println("IBD1 to IBD2:");
			System.out.println("\t" + f.v1 + "\n\t\t<--> " + f.v2);
		}
	}
}
