package lib.blocks.models;

import java.util.*;

import lib.blocks.models.IBDInstances.*;

public abstract class ComposableModel {
	private boolean isFinished;
	private IBDInstances ibdInstances;
	
	public final ModelLib lib;
	
	public ComposableModel(ModelLib lib) {
		this.lib = lib;
		
		isFinished = false;
		ibdInstances = new IBDInstances(lib);
	}
	
	public Collection<IBD1Instance> getIBD1Instances() {
		return ibdInstances.getIBD1Instances();
	}
	
	public Collection<IBD2Instance> getIBD2Instances() {
		return ibdInstances.getIBD2Instances();
	}
	
	public Set<IBD1Port> getIBD1Ports() {
		Set<IBD1Port> result = new HashSet<IBD1Port>();
		
		for (IBD1Instance ibd1 : getIBD1Instances()) {
			result.addAll(ibd1.getPortPerName().values());
		}
		
		return Collections.unmodifiableSet(result);
	}
	
	public Set<IBD2Port> getIBD2Ports() {
		Set<IBD2Port> result = new HashSet<IBD2Port>();
		
		for (IBD2Instance ibd2 : getIBD2Instances()) {
			result.addAll(ibd2.getPortPerId().values());
		}
		
		return Collections.unmodifiableSet(result);
	}
	
	public Set<IBD1Flow> getIBD1Flows() {
		return ibdInstances.getIBD1Flows();
	}
	
	public Set<IBD2Flow> getIBD2Flows() {
		return ibdInstances.getIBD2Flows();
	}
	
	public Set<IBD1ToIBD2Flow> getIBD1ToIBD2Flows() {
		return ibdInstances.getIBD1ToIBD2Flows();
	}
	
	private void confirmUnfinished() {
		if (isFinished) {
			throw new Error("Cannot add elements to a finished model!");
		}
	}
	
	public boolean isFinished() {
		return isFinished;
	}
	
	public void add(String instanceName, Class<?> instanceTypeClz) {
		add(instanceName, -1, lib.get(instanceTypeClz));
	}
	
	public void add(String instanceName, Block instanceType) {
		add(instanceName, -1, instanceType);
	}
	
	public void add(String name, int index, Block instanceType) {
		confirmUnfinished();
		ibdInstances.addIBDInstance(new NameIndexId(name, index), instanceType);
	}
	
	public List<String> getUnderspecificationMsgs() {
		List<String> result = new ArrayList<String>();
		
		for (IBD1Instance ibd1 : getIBD1Instances()) {
			if (!ibd1.isInstantiated()) {
				result.add("IBD1 " + ibd1.id + " has not been instantiated!");
			}
		}
		
		for (IBD2Instance ibd2 : getIBD2Instances()) {
			if (!ibd2.isInstantiated()) {
				result.add("IBD2 " + ibd2.id + " has not been instantiated!");
			}
		}
		
		return result;
	}
	
	public void finish() {
//		ibdInstances.print();
//		System.exit(0);
		
		List<String> msgs = getUnderspecificationMsgs();
		
		if (msgs.size() > 0) {
			String msg = "Underspecified model:";
			
			for (String m : msgs) {
				msg += "\n\t\t" + m;
			}
			
			throw new Error(msg);
		}
		
		isFinished = true;
	}
	
	public IBD1Instance get(String ibd1Name) {
		return get(new NameIndexId(ibd1Name, -1));
	}
	
	public IBD1Instance get(NameIndexId ibd1Id) {
		for (IBD1Instance ibd1 : getIBD1Instances()) {
			if (ibd1.id.equals(ibd1Id)) {
				return ibd1;
			}
		}
		
		String msg = "Could not find IBD1 named \"" + ibd1Id + "\" among";
		
		for (IBD1Instance ibd1 : getIBD1Instances()) {
			msg += "\n\t\tIBD1 " + ibd1.id;
		}
		
		throw new Error(msg);
	}
}
