package lib.blocks.ibd2;

import java.lang.reflect.Field;

import lib.blocks.ibd1.IBD1Target;
import lib.blocks.models.NameIndexId;
import lib.utils.ClassUtils;

public abstract class IBD2Target {
	public abstract void connectFlows();
	
	public final void disconnectAllFlows() {
		for (Field f : ClassUtils.getAllFields(getClass())) {
			disconnectFlow(f);
		}
	}
	
	private void disconnectFlow(Field f) {
		if (f.getType().isArray()) {
			Class<?> compType = f.getType().getComponentType();
			
			if (InterfacePort.class.isAssignableFrom(compType)) {
				InterfacePort[] values = getValues(this, f, InterfacePort.class);
				
				for (int portIndex = 0; portIndex < values.length; portIndex++) {
					values[portIndex].disconnectAll();
				}
				
				return;
			}
			
			if (IBD1Target.class.isAssignableFrom(compType)) {
				IBD1Target[] values = getValues(this, f, IBD1Target.class);
				
				for (int memberIndex = 0; memberIndex < values.length; memberIndex++) {
					values[memberIndex].disconnectAllFlows();
				}
				
				return;
			}
			
			if (IBD2Target.class.isAssignableFrom(compType)) {
				IBD2Target[] values = getValues(this, f, IBD2Target.class);
				
				for (int memberIndex = 0; memberIndex < values.length; memberIndex++) {
					values[memberIndex].disconnectAllFlows();
				}
				
				return;
			}
		} else {
			if (InterfacePort.class.isAssignableFrom(f.getType())) {
				InterfacePort ip = getValue(this, f, InterfacePort.class);
				ip.disconnectAll();
				return;
			}
			
			if (IBD1Target.class.isAssignableFrom(f.getType())) {
				IBD1Target comp = getValue(this, f, IBD1Target.class);
				comp.disconnectAllFlows();
				return;
			}
			
			if (IBD2Target.class.isAssignableFrom(f.getType())) {
				IBD2Target comp = getValue(this, f, IBD2Target.class);
				comp.disconnectAllFlows();
				return;
			}
		}
	}
	
	public final NameIndexId getCompId(Object comp) {
		for (Field f : ClassUtils.getAllFields(getClass())) {
			NameIndexId id = getNameIndexId(comp, f);
			
			if (id != null) {
				return id;
			}
		}
		
		throw new Error("Should not happen; could not find component!");
	}
	
	private NameIndexId getNameIndexId(Object comp, Field f) {
		if (f.getType().isArray()) {
			Class<?> compType = f.getType().getComponentType();
			
			if (IBD1Target.class.isAssignableFrom(compType)) {
				IBD1Target[] values = getValues(this, f, IBD1Target.class);
				
				for (int memberIndex = 0; memberIndex < values.length; memberIndex++) {
					if (values[memberIndex] == comp) {
						return new NameIndexId(f.getName(), memberIndex);
					}
				}
				
				return null;
			}
			
			if (IBD2Target.class.isAssignableFrom(compType)) {
				IBD2Target[] values = getValues(this, f, IBD2Target.class);
				
				for (int memberIndex = 0; memberIndex < values.length; memberIndex++) {
					if (values[memberIndex] == comp) {
						return new NameIndexId(f.getName(), memberIndex);
					}
				}
				
				return null;
			}
		} else {
			if (IBD1Target.class.isAssignableFrom(f.getType())) {
				IBD1Target value = getValue(this, f, IBD1Target.class);
				
				if (value == comp) {
					return new NameIndexId(f.getName(), -1);
				}
				
				return null;
			}
			
			if (IBD2Target.class.isAssignableFrom(f.getType())) {
				IBD2Target value = getValue(this, f, IBD2Target.class);
				
				if (value == comp) {
					return new NameIndexId(f.getName(), -1);
				}
				
				return null;
			}
		}
		
		//Not the right type of field:
		return null;
	}
	
	private static <T> T getValue(Object instance, Field f, Class<T> clz) {
		try {
			return clz.cast(f.get(instance));
		} catch (IllegalArgumentException | IllegalAccessException e) {
			throw new Error(e);
		}
	}
	
	@SuppressWarnings("unchecked")
	private static <T> T[] getValues(Object instance, Field f, Class<T> clz) {
		try {
			return (T[])clz.cast(f.get(instance));
		} catch (IllegalArgumentException | IllegalAccessException e) {
			throw new Error(e);
		}
	}
}
