package lib.blocks.constraints;

import lib.blocks.common.IncompatibilityException;
import lib.blocks.ibd1.IBD1Target;
import lib.blocks.ibd2.IBD2Interface;
import lib.blocks.ibd2.IBD2Target;
import lib.blocks.models.*;
import lib.blocks.models.IBDInstances.*;

/**
 * This constraint defines the superblock of a block (if any).
 * If the superblock is set to null, a block must NOT have a user-defined superblock.
 */
public class GeneralizationConstraint extends BlockConstraint {
	private Block superBlock;
	
	private GeneralizationConstraint(Class<?> sourceClz) {
		sources.add(sourceClz);
	}
	
	private GeneralizationConstraint(GeneralizationConstraint c1, GeneralizationConstraint c2) {
		super(c1, c2);
	}
	
	@Override
	public String toString() {
		if (superBlock != null) {
			return "<Gen> " + superBlock.id;
		}
		
		return "<Gen> Object";
	}
	
	@Override
	public BlockConstraint createCopy() {
		GeneralizationConstraint result = new GeneralizationConstraint(null);
		result.sources.add(sources);
		result.superBlock = superBlock;
		return result;
	}
	
	/**
	 * (Never null.)
	 */
	public Block getSuperBlock() {
		return superBlock;
	}
	
	public static void fromDiagram(ModelLib lib, Class<?> clz) throws IncompatibilityException {
		switch (SourceType.get(clz)) {
			case BDD:
				fromBDD(lib, clz);
				break;
			case IBD1:
				fromIBD1(lib, clz);
				break;
			case IBD2:
				fromIBD2(lib, clz);
				break;
			case IBD2_INTERFACE:
				fromIBD2Interface(lib, clz);
				break;
		}
	}
	
	private static void fromBDD(ModelLib lib, Class<?> clz) throws IncompatibilityException {
		if (clz.getSuperclass() != null && clz.getSuperclass() != Object.class) {
			GeneralizationConstraint a = new GeneralizationConstraint(clz);
			a.superBlock = lib.get(clz.getSuperclass());
			lib.declareConstraint(clz, a);
		}
	}
	
	private static void fromIBD1(ModelLib lib, Class<?> clz) throws IncompatibilityException {
		GeneralizationConstraint a = new GeneralizationConstraint(clz);
		
		if (clz.getSuperclass() != IBD1Target.class) {
			a.superBlock = lib.get(clz.getSuperclass());
		}
		
		lib.declareConstraint(clz, a);
	}
	
	private static void fromIBD2(ModelLib lib, Class<?> clz) throws IncompatibilityException {
		GeneralizationConstraint a = new GeneralizationConstraint(clz);
		
		if (clz.getSuperclass() != IBD2Target.class) {
			a.superBlock = lib.get(clz.getSuperclass());
		}
		
		lib.declareConstraint(clz, a);
	}
	
	private static void fromIBD2Interface(ModelLib lib, Class<?> clz) throws IncompatibilityException {
		GeneralizationConstraint a = new GeneralizationConstraint(clz);
		
		if (clz.getSuperclass() != IBD2Interface.class) {
			a.superBlock = lib.get(clz.getSuperclass());
		}
		
		lib.declareConstraint(clz, a);
	}
	
	public static GeneralizationConstraint combine(GeneralizationConstraint a1, GeneralizationConstraint a2) throws IncompatibleConstraintsException {
		GeneralizationConstraint result = new GeneralizationConstraint(a1, a2);
		result.superBlock = getSameValue(a1, a2, a1.superBlock, a2.superBlock);
		return result;
	}
	
	@Override
	public void checkConformance(Model model, IBD1Instance ibd1) throws ViolatedConstraintException {
		//We do not care anymore about inheritance once a model is underway.
	}
	
	@Override
	public void checkConformance(Model model, IBD2Instance ibd2) throws ViolatedConstraintException {
		//We do not care anymore about inheritance once a model is underway.
	}
}
