package lib.blocks.constraints;

import java.lang.reflect.Field;
import java.util.*;

import lib.blocks.bdds.*;
import lib.blocks.common.*;
import lib.blocks.models.*;
import lib.blocks.models.IBDInstances.*;
import lib.utils.*;

/**
 * A block with this aspect in a completed system MUST have a 'communication
 * line' that corresponds with this aspect.
 */
public class AssociationConstraint extends BlockConstraint {
	private String name;
	private Multiplicity mult;
	private Set<FlowSpecExpr> flowSpecExprs;
//	private AssocReq assocReq;
	private Block otherBlock;
	private String conjugateFieldName;
	
//	private static enum AssocReq {
//		UNIQUE,
//		NOT_UNIQUE,
//		UNKNOWN,
//	}
	
	private AssociationConstraint(Field sourceField) {
		mult = new Multiplicity(sourceField);
		flowSpecExprs = new HashSet<FlowSpecExpr>();
		sources.add(sourceField);
	}
	
	private AssociationConstraint(AssociationConstraint c1, AssociationConstraint c2) throws IncompatibleMultiplicitiesException {
		super(c1, c2);
		
		mult = new Multiplicity(c1.mult, c2.mult);
		flowSpecExprs = new HashSet<FlowSpecExpr>();
	}
	
	@Override
	public String toString() {
		String result = "<Assoc>";
		
//		switch (assocReq) {
//			case UNIQUE:
//				result = "<UniqueAssoc>";
//				break;
//			case NOT_UNIQUE:
//				result = "<Assoc>";
//				break;
//			default:
//				result = "<Assoc/UniqueAssoc>";
//				break;
//		}
		
		result += " " + name + ": ";
		
		if (otherBlock != null) {
			result += otherBlock.id;
		} else {
			result += "UnknownType";
		}
		
		result += mult.toString();
		
		if (flowSpecExprs.size() > 0) {
			Iterator<FlowSpecExpr> q = flowSpecExprs.iterator();
			result += " { " + q.next().toString();
			
			while (q.hasNext()) {
				result += ", " + q.next().toString();
			}
			
			result += " }";
		}
		
		if (conjugateFieldName != null) {
			result += " <= " + conjugateFieldName;
		}
		
		return result;
	}
	
	@Override
	public BlockConstraint createCopy() {
		AssociationConstraint result = new AssociationConstraint(null);
		result.sources.add(sources);
		result.name = name;
		result.mult = mult;
		result.flowSpecExprs.addAll(flowSpecExprs);
//		result.assocReq = assocReq;
		result.otherBlock = otherBlock;
		result.conjugateFieldName = conjugateFieldName;
		return result;
	}
	
	public String getName() {
		return name;
	}
	
	public Multiplicity getMult() {
		return mult;
	}
	
	public Set<FlowSpecExpr> getFlowSpecExprs() {
		return Collections.unmodifiableSet(flowSpecExprs);
	}
	
	/**
	 * Blocks must have exactly 0 or 2 associations with 'isUniqueAssoc' set to
	 * 'true'. In a completed system, there must exist up to 1 instance of the block
	 * for each permutation of 'other block' instances. That is, if an instance of
	 * this block X associates with A1 and with A2, then there cannot be another
	 * instance of this block Y that also associates with A1 and A2 (but it could
	 * associate with A2 and A1).
	 */
//	public boolean isUniqueAssoc() {
//		return assocReq == AssocReq.UNIQUE;
//	}
	
	public Block getOtherBlock() {
		return otherBlock;
	}
	
	public String getConjugateFieldName() {
		return conjugateFieldName;
	}
	
	public static void fromDiagram(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		switch (SourceType.get(clz)) {
			case BDD:
				fromBDD(lib, clz);
				break;
			case IBD1:
				// (IBD1s do not produce associations.)
				break;
			case IBD2:
				fromIBD2(lib, clz);
				break;
			case IBD2_INTERFACE:
				// (IBD interfaces do not produce ASAL ports.)
				break;
		}
	}
	
	private static void fromBDD(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		BDD bddInstance = BDD.class.cast(ClassUtils.createStdInstance(clz.getDeclaringClass()));
		bddInstance.initBidirMap(lib, clz);
		bddInstance.initBidirs();
		
		for (Field f : ClassUtils.getDeclaredFields(clz)) {
			ParameterizedField pf = ParameterizedField.obtain(f, Assoc.class, Comp.class, Aggr.class);
			
			if (Assoc.class.isAssignableFrom(pf.parameterizedType)) {
				BidirMap.Bidir bidir = bddInstance.getBidirMap().get(f.getName());
				
				if (bidir == null) {
					throw new Error("Association \"" + f.getName() + "\" of block \"" + clz.getCanonicalName() + "\" should be bidirectionally connected to another association!");
				}
				
				fromBDDBidir(lib, clz, bidir);
			}
		}
		
		//TODO are all assocs connected by bidirs?
	}
	
//	private static void fromExposedIBD2(ModelLib lib, Class<?> clz) throws IncompatibilityException {
//		for (Field f : ClassUtils.getDeclaredFields(clz)) {
//			fromExposedIBD2Field(lib, clz, f);
//		}
//	}
//	
//	private static void fromExposedIBD2Field(ModelLib lib, Class<?> clz, Field f) throws IncompatibilityException {
//		if (InterfacePort.class.isAssignableFrom(f.getType())) {
//			InterfacePort ip = FieldUtils.getValue(f, InterfacePort.class);
//			
//			AssociationConstraint a = new AssociationConstraint(f);
//			a.name = f.getName();
//			a.assocReq = AssocReq.UNKNOWN;
//			a.mult = Multiplicity.createUnknownDotDotOne(f);
//			a.otherBlock = null; //Unknown!
//			a.conjugateFieldName = null; //Unknown!
//			
//			if (ip.flowSpecExpr != null) {
//				a.flowSpecExprs.add(ip.flowSpecExpr);
//			}
//			
//			lib.declareConstraint(clz, a);
//			return;
//		}
//		
//		if (InterfacePorts.class.isAssignableFrom(f.getType())) {
//			InterfacePorts ips = FieldUtils.getValue(f, InterfacePorts.class);
//			
//			AssociationConstraint a = new AssociationConstraint(f);
//			a.name = f.getName();
//			a.assocReq = AssocReq.UNKNOWN;
//			
//			if (!ips.hasAgnosticSize) {
//				a.mult = Multiplicity.createFromOccurringCount(f, ips.count);
//			}
//			
//			a.otherBlock = null; //Unknown!
//			a.conjugateFieldName = null; //Unknown!
//			
//			if (ips.flowSpecExpr != null) {
//				a.flowSpecExprs.add(ips.flowSpecExpr);
//			}
//			
//			lib.declareConstraint(clz, a);
//			return;
//		}
//	}
	
	private static void fromIBD2(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		Block block = lib.get(clz);
		
		try {
			for (ConnectionPt cp : ConnectionPts.getConnectionPts(block, clz)) {
				fromIBD2ConnectionPt(lib, clz, cp);
			}
		} catch (ConnectionPtException e) {
			throw new Error("Field " + e.pt.field.getName() + " in diagram " + clz.getCanonicalName(), e);
		}
	}
	
	private static void fromIBD2ConnectionPt(ModelLib lib, Class<?> clz, ConnectionPt cp) throws IncompatibilityException {
		if (cp instanceof ConnectionPt.ExposedPt) {
			ConnectionPt.ExposedPt ep = (ConnectionPt.ExposedPt)cp;
			
			AssociationConstraint a = new AssociationConstraint(ep.field);
			a.name = ep.field.getName();
			
//			if (ep.getAssocEndTargetId() != null) {
//				a.assocReq = AssocReq.UNIQUE;
//				a.mult = Multiplicity.createOneDotDotOne(ep.field);
//				a.otherBlock = null; //Unknown (generally)!
//				a.conjugateFieldName = null; //Unknown (generally)!
//				
//				if (ep.getFlowSpecExpr() != null) {
//					a.flowSpecExprs.add(ep.getFlowSpecExpr());
//				}
//				
//				lib.declareConstraint(ep.ownerBlock, a);
//				return;
//			}
			
			ConnectionPt.ExposedPt otherSideEp = ep.getOtherSide();
			
			if (otherSideEp != null) {
//				a.assocReq = AssocReq.NOT_UNIQUE;
				
				if (ep.getPortId().index >= 0) {
					a.mult = Multiplicity.createFromOccurringCount(ep.field, ep.getPortId().index + 1);
				} else {
					a.mult = Multiplicity.createUnknownDotDotOne(ep.field);
				}
				
				a.otherBlock = null; //Unknown (generally)!
				a.conjugateFieldName = null; //Unknown (generally)!
				
				if (ep.getFlowSpecExpr() != null) {
					a.flowSpecExprs.add(ep.getFlowSpecExpr());
					a.flowSpecExprs.add(otherSideEp.getFlowSpecExpr().createConjugate());
				}
				
				lib.declareConstraint(ep.ownerBlock, a);
				return;
			}
			
//			a.assocReq = AssocReq.UNKNOWN;
			
			if (ep.getPortId().index >= 0) {
				a.mult = Multiplicity.createFromOccurringCount(ep.field, ep.getPortId().index + 1);
			} else {
				a.mult = Multiplicity.createFromOccurringCount(ep.field, 1);
			}
			
			a.otherBlock = null; //Unknown!
			a.conjugateFieldName = null; //Unknown!
			
			if (ep.getFlowSpecExpr() != null) {
				a.flowSpecExprs.add(ep.getFlowSpecExpr());
			}
			
			lib.declareConstraint(ep.ownerBlock, a);
			return;
		}
		
		if (cp instanceof ConnectionPt.ExposingPt) {
			ConnectionPt.ExposingPt ep = (ConnectionPt.ExposingPt)cp;
			
			AssociationConstraint a = new AssociationConstraint(ep.field);
			a.name = ep.field.getName();
//			a.assocReq = AssocReq.UNKNOWN;
			
			if (ep.getPortId().index >= 0) {
				a.mult = Multiplicity.createFromOccurringCount(ep.field, ep.getPortId().index + 1);
			} else {
				a.mult = Multiplicity.createUnknownDotDotOne(ep.field);
			}
			
			a.otherBlock = null; //Unknown!
			a.conjugateFieldName = null; //Unknown!
			
			if (ep.getFlowSpecExpr() != null) {
				a.flowSpecExprs.add(ep.getFlowSpecExpr());
			}
			
			lib.declareConstraint(ep.ownerBlock, a);
			return;
		}
	}
	
	private static void fromBDDBidir(ModelLib lib, Class<?> clz, BidirMap.Bidir bidir) throws IncompatibilityException {
		if (bidir.getAssocBlock() != null) {
			AssociationConstraint a1 = new AssociationConstraint(bidir.getField1());
			AssociationConstraint a2 = new AssociationConstraint(bidir.getField2());
			AssociationConstraint b1 = new AssociationConstraint(bidir.getField2());
			AssociationConstraint b2 = new AssociationConstraint(bidir.getField1());
			
			a1.name = bidir.getField1().getName();
			a1.mult = Multiplicity.create(bidir.getField1(), bidir.getMinCount1(), bidir.getMaxCount1());
//			a1.assocReq = AssocReq.NOT_UNIQUE;
			a1.otherBlock = bidir.getAssocBlock();
			a1.conjugateFieldName = bidir.getField2().getName();
			
			a2.name = bidir.getField2().getName();
			a2.mult = Multiplicity.createOneDotDotOne(bidir.getField2());
//			a2.assocReq = AssocReq.UNIQUE;
			a2.otherBlock = bidir.getOwningBlock1();
			a2.conjugateFieldName = bidir.getField1().getName();
			
			b2.name = bidir.getField1().getName();
			b2.mult = Multiplicity.createOneDotDotOne(bidir.getField1());
//			b2.assocReq = AssocReq.UNIQUE;
			b2.otherBlock = bidir.getOwningBlock2();
			b2.conjugateFieldName = bidir.getField2().getName();
			
			b1.name = bidir.getField2().getName();
			b1.mult = Multiplicity.create(bidir.getField2(), bidir.getMinCount2(), bidir.getMaxCount2());
//			b1.assocReq = AssocReq.NOT_UNIQUE;
			b1.otherBlock = bidir.getAssocBlock();
			b1.conjugateFieldName = bidir.getField1().getName();
			
			lib.declareConstraint(bidir.getOwningBlock1(), a1);
			lib.declareConstraint(bidir.getAssocBlock(), a2);
			lib.declareConstraint(bidir.getAssocBlock(), b2);
			lib.declareConstraint(bidir.getOwningBlock2(), b1);
		} else {
			AssociationConstraint a = new AssociationConstraint(bidir.getField1());
			AssociationConstraint b = new AssociationConstraint(bidir.getField2());
			
			a.name = bidir.getField1().getName();
			a.mult = Multiplicity.create(bidir.getField1(), bidir.getMinCount1(), bidir.getMaxCount1());
//			a.assocReq = AssocReq.NOT_UNIQUE;
			a.otherBlock = bidir.getOwningBlock2();
			a.conjugateFieldName = bidir.getField2().getName();
			
			b.name = bidir.getField2().getName();
			b.mult = Multiplicity.create(bidir.getField2(), bidir.getMinCount2(), bidir.getMaxCount2());
//			b.assocReq = AssocReq.NOT_UNIQUE;
			b.otherBlock = bidir.getOwningBlock1();
			b.conjugateFieldName = bidir.getField1().getName();
			
			lib.declareConstraint(bidir.getOwningBlock1(), a);
			lib.declareConstraint(bidir.getOwningBlock2(), b);
		}
	}
	
	public static AssociationConstraint combine(AssociationConstraint a1, AssociationConstraint a2, boolean overriding) throws IncompatibilityException {
		if (a1.getName().equals(a2.getName())) {
			try {
				AssociationConstraint result = new AssociationConstraint(a1, a2);
				result.name = a1.getName();
				result.mult = new Multiplicity(a1.mult, a2.mult);
//				result.assocReq = getCompatibleValue(a1, a2, a1.assocReq, a2.assocReq, AssocReq.UNKNOWN);
				result.conjugateFieldName = getCompatibleValue(a1, a2, a1.conjugateFieldName, a2.conjugateFieldName, null);
				result.flowSpecExprs.addAll(a1.flowSpecExprs);
				result.flowSpecExprs.addAll(a2.flowSpecExprs);
				
				if (overriding) {
					if (a1.otherBlock == null) {
						result.otherBlock = a2.otherBlock;
					} else {
						if (a2.otherBlock == null) {
							result.otherBlock = a1.otherBlock;
						} else {
							if (a2.otherBlock.getLineage().contains(a1.otherBlock)) {
								result.otherBlock = a2.otherBlock;
							} else {
								incompatibleConstraints(a1, a2);
							}
						}
					}
				} else {
					result.otherBlock = getCompatibleValue(a1, a2, a1.otherBlock, a2.otherBlock, null);
				}
				
//				if (result.assocReq == AssocReq.UNIQUE) {
//					result.mult = new Multiplicity(result.mult, Multiplicity.createOneDotDotOne(null));
//				}
				
				return result;
			} catch (IncompatibleMultiplicitiesException e) {
				throw new IncompatibleConstraintsException(a1, a2, e);
			}
		}
		
		return null;
	}
	
	@Override
	public void checkConformance(Model model, IBD1Instance ibd1) throws ViolatedConstraintException {
		// IBD1s do not have associations!
	}
	
	@Override
	public void checkConformance(Model model, IBD2Instance ibd2) throws ViolatedConstraintException {
		Set<Integer> encounteredIndices = new HashSet<Integer>();
		
		for (IBD2Port port : ibd2.getPortPerId().values()) {
			if (port.id.name.equals(name)) {
				encounteredIndices.add(port.id.index);
			}
		}
		
		if (mult.getMaxCount() == 0) {
			switch (encounteredIndices.size()) {
				case 0:
					if (model.isFinished() && mult.getMinCount() == 0) {
						throw new ViolatedConstraintException(this, ibd2, "Missing association!");
					}
					break;
				case 1:
					int index = encounteredIndices.iterator().next();
					
					if (index != -1) {
						throw new ViolatedConstraintException(this, ibd2, "Association should be un-indexed!");
					}
					break;
				default:
					throw new ViolatedConstraintException(this, ibd2, "Invalid index!");
			}
		} else {
			if (model.isFinished()) {
				if (encounteredIndices.size() == 1) {
					int value = encounteredIndices.iterator().next();
					
					if (value != 0 && value != -1) {
						String msg = "Unindexed association or association with index 0 is missing!";
						msg += "\n\tFound index " + value + "!";
						throw new ViolatedConstraintException(this, ibd2, msg);
					}
				} else {
					for (int index = 0; index < encounteredIndices.size(); index++) {
						if (!encounteredIndices.contains(index)) {
							String msg = "Association with index " + index + " is missing!";
							
							for (Integer idx : encounteredIndices) {
								msg += "\n\tFound index " + idx + "!";
							}
							
							throw new ViolatedConstraintException(this, ibd2, msg);
						}
					}
				}
			}
			
			for (int encounteredIndex : encounteredIndices) {
				if (!mult.isValidIndex(encounteredIndex)) {
					throw new ViolatedConstraintException(this, ibd2, "Association index " + encounteredIndex + " is out of range!");
				}
			}
			
			if (!mult.isValidCount(encounteredIndices.size())) {
				String msg = "Association count " + encounteredIndices.size() + " is invalid!";
				
				for (IBD2Port port : ibd2.getPortPerId().values()) {
					msg += "\n\t\tFound port " + port.id;
				}
				
				throw new ViolatedConstraintException(this, ibd2, msg);
			}
		}
		
//		if (isUniqueAssoc()) {
//			IBD2Instance assoc1 = getUniqueAssoc(model, ibd2, name);
//			IBD2Instance assoc2 = getUniqueAssoc(model, ibd2, conjugateFieldName);
//			
//			if (assoc1 == null && model.isFinished()) {
//				throw new ViolatedConstraintException(this, ibd2, "Unique association is unassigned!");
//			}
//			
//			if (assoc2 == null && model.isFinished()) {
//				throw new ViolatedConstraintException(this, ibd2, "Unique association is unassigned!");
//			}
//			
//			if (assoc1 != null && assoc2 != null) {
//				Map<IBD2Instance, IBD2Instance> assoc1PerInstance = new HashMap<IBD2Instance, IBD2Instance>();
//				Map<IBD2Instance, IBD2Instance> assoc2PerInstance = new HashMap<IBD2Instance, IBD2Instance>();
//				getUniqueAssocs(model, ibd2.getType(), assoc1PerInstance, assoc2PerInstance);
//				assoc1PerInstance.remove(ibd2);
//				assoc2PerInstance.remove(ibd2);
//				
//				for (IBD2Instance i : assoc1PerInstance.keySet()) {
//					IBD2Instance a1 = assoc1PerInstance.get(i);
//					IBD2Instance a2 = assoc2PerInstance.get(i);
//					
//					if (a1 == assoc1 && a2 == assoc2) {
//						throw new ViolatedConstraintException(this, ibd2, "Unique association constraint is violated!");
//					}
//				}
//			}
//		}
	}
	
	/**
	 * Returns information about ALL instances of an association class that exist in the model.
	 * Information consists of which two instances are related by the association class.
	 */
	private void getUniqueAssocs(Model model, Block type, Map<IBD2Instance, IBD2Instance> assoc1PerInstance, Map<IBD2Instance, IBD2Instance> assoc2PerInstance) {
//		if (!isUniqueAssoc()) {
//			return;
//		}
		
		if (conjugateFieldName == null) {
			return;
		}
		
		for (IBD2Instance i : model.getIBD2Instances()) {
			if (i.getType().getLineage().contains(type)) {
				IBD2Instance assoc1 = getUniqueAssoc(model, i, name);
				IBD2Instance assoc2 = getUniqueAssoc(model, i, conjugateFieldName);
				
				//Neither of the assocs should be null.
				//If this is the case, we handle it when
				//we check the constraint for the IBD2 instance in question:
				if (assoc1 != null && assoc2 != null) {
					assoc1PerInstance.put(i, assoc1);
					assoc2PerInstance.put(i, assoc2);
				}
			}
		}
	}
	
	private IBD2Instance getUniqueAssoc(Model model, IBD2Instance i1, String assocName) {
		for (IBD2Port p1 : i1.getPortPerId().values()) {
			if (p1.id.name.equals(assocName)) {
				IBD2Port p2 = model.getConjugateIBD2Port(p1);
				
				if (p2 != null) {
					return p2.owner;
				}
			}
		}
		
		return null;
	}
}
