package lib.blocks.constraints;

import java.lang.reflect.*;

import lib.blocks.bdds.*;
import lib.blocks.common.*;
import lib.blocks.ibd1.IBD1Target;
import lib.blocks.ibd2.IBD2Target;
import lib.blocks.models.*;
import lib.blocks.models.IBDInstances.*;
import lib.utils.*;

/**
 * This constraint specifies that a block has component of a specific type, or
 * an array of components of a specific type.
 */
public class AggregationConstraint extends BlockConstraint {
	private String name;
	private Multiplicity mult;
	private CompReq compReq;
	private Block otherBlock;
	
	private static enum CompReq {
		COMP,
		NOT_COMP,
		UNKNOWN
	}
	
	private AggregationConstraint(Field sourceField) {
		mult = new Multiplicity(sourceField);
		sources.add(sourceField);
	}
	
	private AggregationConstraint(AggregationConstraint c1, AggregationConstraint c2) throws IncompatibleMultiplicitiesException {
		super(c1, c2);
		
		mult = new Multiplicity(c1.mult, c2.mult);
	}
	
	@Override
	public String toString() {
		String result = name + ": " + otherBlock.id + mult;
		
		switch (compReq) {
			case COMP:
				return "<Comp> " + result;
			case NOT_COMP:
				return "<Aggr> " + result;
			default:
				return "<Aggr/Comp> " + result;
		}
	}
	
	@Override
	public BlockConstraint createCopy() {
		AggregationConstraint result = new AggregationConstraint(null);
		result.sources.add(sources);
		result.name = name;
		result.mult = mult;
		result.compReq = compReq;
		result.otherBlock = otherBlock;
		return result;
	}
	
	public String getName() {
		return name;
	}
	
	public Multiplicity getMult() {
		return mult;
	}
	
	/**
	 * If false, aggregated blocks are NEVER implicitly instantiated,
	 * even if there is one obvious IBD that can be used for instantiation!
	 */
	public boolean isComp() {
		return compReq == CompReq.COMP;
	}
	
	public Block getOtherBlock() {
		return otherBlock;
	}
	
	public static void fromDiagram(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		switch (SourceType.get(clz)) {
			case BDD:
				fromBDD(lib, clz);
				break;
			case IBD1:
				fromIBD1(lib, clz);
				break;
			case IBD2:
				fromIBD2(lib, clz);
				break;
			case IBD2_INTERFACE:
				// (IBD interfaces do not produce ASAL ports.)
				break;
		}
	}
	
	private static void fromBDD(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		for (Field f : ClassUtils.getDeclaredFields(clz)) {
			fromBDDField(lib, clz, f);
		}
	}
	
	private static void fromIBD1(ModelLib lib, Class<?> clz) {
		// Do nothing.
	}
	
	private static void fromIBD2(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		for (Field f : ClassUtils.getDeclaredFields(clz)) {
			fromIBD2Field(lib, clz, f);
		}
	}
	
	private static void fromBDDField(ModelLib lib, Class<?> clz, Field f) throws IncompatibilityException, ReflectionException {
		ParameterizedField pf = ParameterizedField.obtain(f, Assoc.class, Comp.class, Aggr.class);
		
		if (Comp.class.isAssignableFrom(pf.parameterizedType)) {
			Comp<?> r = FieldUtils.getValue(f, Comp.class);
			
			AggregationConstraint a = new AggregationConstraint(f);
			a.name = f.getName();
			a.mult = Multiplicity.create(f, r.minCount, r.maxCount);
			a.compReq = CompReq.COMP;
			a.otherBlock = lib.get(pf.parameterType);
			lib.declareConstraint(clz, a);
			return;
		}
		
		if (Aggr.class.isAssignableFrom(pf.parameterizedType)) {
			Aggr<?> r = FieldUtils.getValue(f, Aggr.class);
			
			AggregationConstraint a = new AggregationConstraint(f);
			a.name = f.getName();
			a.mult = Multiplicity.create(f, r.minCount, r.maxCount);
			a.compReq = CompReq.NOT_COMP;
			a.otherBlock = lib.get(pf.parameterType);
			lib.declareConstraint(clz, a);
			return;
		}
	}
	
	private static void fromIBD2Field(ModelLib lib, Class<?> clz, Field f) throws IncompatibilityException, ReflectionException {
		if (f.getType().isArray()) {
			Class<?> compType = f.getType().getComponentType();
			
			if (IBD1Target.class.isAssignableFrom(compType) || IBD2Target.class.isAssignableFrom(compType)) {
				Object[] array = FieldUtils.getValues(f, Object.class);
				
				if (array == null) {
					throw new Error("Array \"" + f.getName() + "\" of class \"" + clz.getCanonicalName() + "\" is not initialized!");
				}
				
				for (int index = 0; index < array.length; index++) {
					if (array[index] == null) {
						throw new Error("Element " + index + " in array \"" + f.getName() + "\" of class \"" + clz.getCanonicalName() + "\" is not initialized!");
					}
				}
				
				AggregationConstraint a = new AggregationConstraint(f);
				a.name = f.getName();
				a.mult = Multiplicity.createFromOccurringCount(f, array.length);
				a.compReq = CompReq.UNKNOWN;
				a.otherBlock = lib.get(f.getType());
				lib.declareConstraint(clz, a);
			}
		} else {
			if (IBD1Target.class.isAssignableFrom(f.getType()) || IBD2Target.class.isAssignableFrom(f.getType())) {
				AggregationConstraint a = new AggregationConstraint(f);
				a.name = f.getName();
				a.mult = Multiplicity.createUnknownDotDotOne(f);
				a.compReq = CompReq.UNKNOWN;
				a.otherBlock = lib.get(f.getType());
				lib.declareConstraint(clz, a);
			}
		}
	}
	
	public static AggregationConstraint combine(AggregationConstraint a1, AggregationConstraint a2) throws IncompatibleConstraintsException {
		if (a1.getName().equals(a2.getName())) {
			try {
				AggregationConstraint result = new AggregationConstraint(a1, a2);
				result.name = a1.getName();
				result.mult = new Multiplicity(a1.mult, a2.mult);
				result.compReq = getCompatibleValue(a1, a2, a1.compReq, a2.compReq, CompReq.UNKNOWN);
				result.otherBlock = getCompatibleValue(a1, a2, a1.otherBlock, a2.otherBlock, null);
				return result;
			} catch (IncompatibleMultiplicitiesException e) {
				throw new IncompatibleConstraintsException(a1, a2, e);
			}
		}
		
		return null;
	}
	
	@Override
	public void checkConformance(Model model, IBD1Instance ibd1) throws ViolatedConstraintException {
		// IBD1s do not have aggregations!
	}
	
	@Override
	public void checkConformance(Model model, IBD2Instance ibd2) throws ViolatedConstraintException {
//		Set<Model2.IBD> elemChildren = ibd2.getChildIBD1s();
//		
//		if (mult.getMax() >= 0 && elemChildren.size() > mult.getMax()) {
//			String msg = "Model element \"" + elem.name + "\" of type \"\" can have at most " + mult.getMax() + " components of type \"" + otherBlock.id + "\", but found";
//			
//			for (Model.Elem n : elemChildren) {
//				msg += "\n\t" + n;
//			}
//			
//			problems.add(msg);
//		}
//		
//		if (model.isFinished() && elemChildren.size() < mult.getMin()) {
//			if (mult.getMin() == 0) {
//				String msg = "Model element \"" + elem.name + "\" of type \"\" must have at least " + mult.getMin() + " components of type \"" + otherBlock.id + "\", but none were found!";
//				problems.add(msg);
//			}
//			
//			if (mult.getMin() > 0) {
//				String msg = "Model element \"" + elem.name + "\" of type \"\" must have at least " + mult.getMin() + " components of type \"" + otherBlock.id + "\", but found";
//				
//				for (Model.Elem n : elemChildren) {
//					msg += "\n\t" + n;
//				}
//				
//				problems.add(msg);
//			}
//		}
		
		// Subcomponents are renamed too GLOBALLY UNIQUE NAMES... so we cannot check them anymore!!
	}
}
