package lib.blocks.constraints;

import java.lang.reflect.*;

import lib.asal.ASALDataType;
import lib.blocks.common.*;
import lib.blocks.ibd1.*;
import lib.blocks.models.*;
import lib.blocks.models.IBDInstances.*;
import lib.utils.*;

/**
 * IMPORTANT
 * This constraint is necessary for checking whether overridden primitive ports are compatible (= the same).
 */
public class ASALPortConstraint extends BlockConstraint {
	private String name;
	private ASALDataType type;
	private Dir dir;
	
	private ASALPortConstraint(Field sourceField) {
		name = null;
		dir = null;
		type = null;
		
		sources.add(sourceField);
	}
	
	private ASALPortConstraint(ASALPortConstraint c1, ASALPortConstraint c2) {
		super(c1, c2);
		
		name = null;
		dir = null;
		type = null;
	}
	
	@Override
	public String toString() {
		String result;
		
		switch (dir) {
			case IN:
				result = "<InPort>";
				break;
			default:
				result = "<OutPort>";
				break;
		}
		
		result += " " + name + ": " + type.name;
		
		return result;
	}
	
	@Override
	public BlockConstraint createCopy() {
		ASALPortConstraint result = new ASALPortConstraint(null);
		result.sources.add(sources);
		result.name = name;
		result.type = type;
		result.dir = dir;
		return result;
	}
	
	public String getName() {
		return name;
	}
	
	public ASALDataType getType() {
		return type;
	}
	
	public Dir getDir() {
		return dir;
	}
	
	public static void fromDiagram(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		switch (SourceType.get(clz)) {
			case BDD:
				// (BDDs do not produce ASAL ports.)
				break;
			case IBD1:
				fromIBD1(lib, clz);
				break;
			case IBD2:
				// (IBDs do not produce ASAL ports.)
				break;
			case IBD2_INTERFACE:
				// (IBD interfaces do not produce ASAL ports.)
				break;
		}
	}
	
	private static void fromIBD1(ModelLib lib, Class<?> clz) throws IncompatibilityException, ReflectionException {
		for (Field f : ClassUtils.getAllFields(clz)) {
			fromIBD1Field(lib, clz, f);
		}
	}
	
	private static void fromIBD1Field(ModelLib lib, Class<?> clz, Field f) throws IncompatibilityException, ReflectionException {
		ParameterizedField pf = ParameterizedField.obtain(f, InPort.class, OutPort.class);
		ASALPortConstraint a = new ASALPortConstraint(f);
		
		a.name = f.getName();
		a.type = ASALDataType.get(pf.parameterType);
		a.dir = InPort.class.isAssignableFrom(pf.parameterizedType) ? Dir.IN : Dir.OUT;
		
		lib.declareConstraint(clz, a);
	}
	
	public static ASALPortConstraint combine(ASALPortConstraint a1, ASALPortConstraint a2, boolean overriding) throws IncompatibilityException {
		if (a1.getName().equals(a2.getName())) {
			ASALPortConstraint result = new ASALPortConstraint(a1, a2);
			result.name = a1.getName();
			result.type = getSameValue(a1, a2, a1.type, a2.type);
			result.dir = getSameValue(a1, a2, a1.dir, a2.dir);
			return result;
		}
		
		return null;
	}
	
	@Override
	public void checkConformance(Model model, IBD1Instance ibd1) throws ViolatedConstraintException {
		// We already know that IBD1s have the required ports!
	}
	
	@Override
	public void checkConformance(Model model, IBD2Instance ibd2) throws ViolatedConstraintException {
		// We already know that IBD2s have the required ports!
	}
}
