package lib.blocks.common;

import java.lang.reflect.*;

import lib.utils.Sources;

/**
 * Contains the definition of a multiplicity.
 * Builds up iteratively, starting with "0..*".
 */
public class Multiplicity {
	/**
	 * If not null, the modeler has manually defined the lowest index that is optional.
	 * If above {@link #highestPossibleIndexMax} (if not null), there is no optional index.
	 */
	private Integer lowestOptionalIndex;
	
	/**
	 * We know that the highest possible index has at least this value,
	 * because we encountered an instance of that value.
	 * Only used if {@link #highestPossibleIndexMax} is null!!
	 */
	private int highestPossibleIndexMin;
	
	/**
	 * If not null, the modeler has manually defined the highest possible index to have this value.
	 */
	private Integer highestPossibleIndexMax;
	
	/**
	 * Which classes/fields are the origin of this multiplicity?
	 */
	public final Sources sources;
	
	public Multiplicity(Field sourceField) {
		sources = new Sources();
		sources.add(sourceField);
		
		lowestOptionalIndex = null;
		highestPossibleIndexMin = 0;
		highestPossibleIndexMax = null;
	}
	
	public Multiplicity(Multiplicity m1, Multiplicity m2) throws IncompatibleMultiplicitiesException {
		sources = new Sources(m1.sources, m2.sources);
		
		lowestOptionalIndex = getCompatibleValue(m1, m2, m1.lowestOptionalIndex, m2.lowestOptionalIndex, null);
		highestPossibleIndexMin = Math.max(m1.highestPossibleIndexMin, m2.highestPossibleIndexMin);
		highestPossibleIndexMax = getCompatibleValue(m1, m2, m1.highestPossibleIndexMax, m2.highestPossibleIndexMax, null);
		
		if (highestPossibleIndexMax != null) {
			if (highestPossibleIndexMin > highestPossibleIndexMax) {
				throw new IncompatibleMultiplicitiesException(m1, m2);
			}
		}
	}
	
	private static <T> T getCompatibleValue(Multiplicity m1, Multiplicity m2, T value1, T value2, T undefinedValue) throws IncompatibleMultiplicitiesException {
		if (value1 == undefinedValue) {
			return value2;
		}
		
		if (value2 == undefinedValue) {
			return value1;
		}
		
		if (value1.equals(value2)) {
			return value1;
		}
		
		throw new IncompatibleMultiplicitiesException(m1, m2);
	}
	
	public int getMinCount() {
		if (lowestOptionalIndex == null) {
			return 0;
		}
		
		if (highestPossibleIndexMax == null || lowestOptionalIndex <= highestPossibleIndexMax) {
			return lowestOptionalIndex + 1;
		}
		
		return highestPossibleIndexMax + 1;
	}
	
	public int getMaxCount() {
		if (highestPossibleIndexMax != null) {
			return highestPossibleIndexMax + 1;
		}
		
		return Integer.MAX_VALUE;
	}
	
	public static Multiplicity createUnknownDotDotOne(Field sourceField) {
		Multiplicity result = new Multiplicity(sourceField);
		result.lowestOptionalIndex = null;
		result.highestPossibleIndexMin = 0;
		result.highestPossibleIndexMax = 0;
		return result;
	}
	
	public static Multiplicity createZeroDotDotOne(Field sourceField) {
		Multiplicity result = new Multiplicity(sourceField);
		result.lowestOptionalIndex = 0;
		result.highestPossibleIndexMin = 0;
		result.highestPossibleIndexMax = 0;
		return result;
	}
	
	public static Multiplicity createOneDotDotOne(Field sourceField) {
		Multiplicity result = new Multiplicity(sourceField);
		result.lowestOptionalIndex = 1;
		result.highestPossibleIndexMin = 0;
		result.highestPossibleIndexMax = 0;
		return result;
	}
	
	public static Multiplicity createFromOccurringCount(Field sourceField, int occurringCount) {
		Multiplicity result = new Multiplicity(sourceField);
		result.lowestOptionalIndex = null;
		result.highestPossibleIndexMin = occurringCount - 1;
		result.highestPossibleIndexMax = null;
		return result;
	}
	
	public static Multiplicity create(Field sourceField, int minCount, int maxCount) {
		Multiplicity result;
		
		switch (minCount) {
			case 0:
				switch (maxCount) {
					case 0:
						throw new Error("Should not happen!");
					case 1:
						return createZeroDotDotOne(sourceField);
					default:
						result = new Multiplicity(sourceField);
						result.lowestOptionalIndex = 0;
						result.highestPossibleIndexMin = maxCount - 1;
						result.highestPossibleIndexMax = maxCount - 1;
						return result;
				}
			case 1:
				switch (maxCount) {
					case 0:
						throw new Error("Should not happen!");
					case 1:
						return createOneDotDotOne(sourceField);
					default:
						result = new Multiplicity(sourceField);
						result.lowestOptionalIndex = 1;
						result.highestPossibleIndexMin = maxCount - 1;
						result.highestPossibleIndexMax = maxCount - 1;
						return result;
				}
			default:
				switch (maxCount) {
					case 0:
						throw new Error("Should not happen!");
					case 1:
						throw new Error("Should not happen!");
					default:
						result = new Multiplicity(sourceField);
						result.lowestOptionalIndex = minCount;
						result.highestPossibleIndexMin = maxCount - 1;
						result.highestPossibleIndexMax = maxCount - 1;
						return result;
				}
		}
	}
	
	public String getDebugStr() {
		String result = "[";
		
		if (lowestOptionalIndex != null) {
			result += lowestOptionalIndex;
		} else {
			result += "(0)";
		}
		
		result += ".." + highestPossibleIndexMin;
		
		if (highestPossibleIndexMax != null) {
			result += ".." + highestPossibleIndexMax;
		} else {
			result += "..*";
		}
		
		result += "]=>" + getSimpleStr();
		return result;
	}
	
	public String getSimpleStr() {
		int maxCount = getMaxCount();
		
		if (maxCount == Integer.MAX_VALUE) {
			return "[" + getMinCount() + "..*]";
		}
		
		return "[" + getMinCount() + ".." + maxCount + "]";
	}
	
	@Override
	public String toString() {
		return getSimpleStr();
	}
	
	public boolean isUnderspecified() {
		return highestPossibleIndexMax == null;
	}
	
	public boolean isValidIndex(int index) {
		if (highestPossibleIndexMax != null) {
			if (highestPossibleIndexMax == 0) {
				return index == -1;
			}
			
			if (highestPossibleIndexMax > 0) {
				return index >= 0 && index <= highestPossibleIndexMax;
			}
		}
		
		return false;
	}
	
	public boolean isValidCount(int count) {
		if (count < getMinCount()) {
			return false;
		}
		
		if (count > getMaxCount()) {
			return false;
		}
		
		return true;
	}
}

