package lib.blocks.common;

import java.util.*;

import lib.blocks.constraints.AssociationConstraint;
import lib.blocks.models.*;
import lib.utils.FilterUtils;

/**
 * Maps each primitive point to record of interaction information, e.g.
 * the names of the interfaces via which the primitive port sends its values.
 * Remember, a primitive port has a unique primitive point for a different context;
 * this means that interaction information is also unique per context. 
 */
public class InteractionMap {
	public static class Interaction {
		private Set<NameIndexId> interactionNames;
		private Set<NameIndexId> oppInteractionNames;
		private Set<ConnectionPt.PrimitivePt> oppPrimPts;
		
		public final ConnectionPt.PrimitivePt primPt;
		
		private Interaction(ConnectionPt.PrimitivePt primPt) {
			this.primPt = primPt;
			
			interactionNames = new HashSet<NameIndexId>();
			oppInteractionNames = new HashSet<NameIndexId>();
			oppPrimPts = new HashSet<ConnectionPt.PrimitivePt>();
		}
		
		public Set<NameIndexId> getInteractionNames() {
			return Collections.unmodifiableSet(interactionNames);
		}
		
		public Set<NameIndexId> getOppInteractionNames() {
			return Collections.unmodifiableSet(oppInteractionNames);
		}
		
		public Set<ConnectionPt.PrimitivePt> getOppPrimPts() {
			return Collections.unmodifiableSet(oppPrimPts);
		}
	}
	
	private Map<ConnectionPt.PrimitivePt, Interaction> interactionPerPrimPt;
	
	public InteractionMap() {
		interactionPerPrimPt = new HashMap<ConnectionPt.PrimitivePt, Interaction>();
	}
	
	public void clear() {
		interactionPerPrimPt.clear();
	}
	
	public Set<ConnectionPt.PrimitivePt> getPrimPts() {
		return Collections.unmodifiableSet(interactionPerPrimPt.keySet());
	}
	
	public Interaction getInteraction(ConnectionPt.PrimitivePt primPt) {
		return interactionPerPrimPt.get(primPt);
	}
	
	private Interaction getOrCreateInteraction(ConnectionPt.PrimitivePt primPt) {
		Interaction result = interactionPerPrimPt.get(primPt);
		
		if (result == null) {
			result = new Interaction(primPt);
			interactionPerPrimPt.put(primPt, result);
		}
		
		return result;
	}
	
	public void add(Block block) {
		for (ConnectionPt pt : block.getContextPts()) {
			if (pt instanceof ConnectionPt.PrimitivePt) {
				ConnectionPt.PrimitivePt pp = (ConnectionPt.PrimitivePt) pt;
				Interaction interaction = getOrCreateInteraction(pp);
				
				for (ConnectionPt connectedPt : pp.getConnectedPts()) {
					if (connectedPt instanceof ConnectionPt.PrimitivePt) {
						ConnectionPt.PrimitivePt pp2 = (ConnectionPt.PrimitivePt)connectedPt;
						interaction.oppPrimPts.add(pp2);
						getOrCreateInteraction(pp2).oppPrimPts.add(pp);
					}
					
					if (connectedPt instanceof ConnectionPt.ExposingPt) {
						ConnectionPt.ExposingPt ep = (ConnectionPt.ExposingPt) connectedPt;
						NameIndexId newElemName = ep.getPortId();
						interaction.interactionNames.add(newElemName);
						
						List<AssociationConstraint> constraints = FilterUtils.filter(ep.ownerBlock.getInheritedConstraints(), AssociationConstraint.class, (c) -> {
							if (c.getName().equals(ep.field.getName())) {
								return c.getOtherBlock() != null && c.getConjugateFieldName() != null;
							}
							
							return false;
						});
						
						for (AssociationConstraint c : constraints) {
							NameIndexId oppElemName = new NameIndexId(c.getConjugateFieldName(), ep.getPortId().index);
							interaction.oppInteractionNames.add(oppElemName);
						}
					}
				}
			}
		}
	}
	
	public void process() {
		for (Map.Entry<ConnectionPt.PrimitivePt, Interaction> entry : interactionPerPrimPt.entrySet()) {
			for (ConnectionPt.PrimitivePt oppPrimPt : entry.getValue().oppPrimPts) {
				getOrCreateInteraction(oppPrimPt).interactionNames.addAll(entry.getValue().oppInteractionNames);
			}
		}
	}
}
