package lib.blocks.common;

import java.util.*;

import lib.blocks.constraints.*;
import lib.blocks.models.Block;
import lib.utils.FilterUtils;

public class FlowSpecExprMap {
	private Map<ConnectionPt.InterfacePt, FlowSpecExprs> exprsPerPt;
	private Map<ConnectionPt.InterfacePt, FlowSpecExprs> reprFlowSpecsPerPt;
	private Set<ConnectionPt.PrimitivePt> primPts;
	private Map<Port, ConnectionPt.InterfacePt> itfPtPerPort;
	private Set<String> userDefinedFlowSpecNames;
	private Set<String> reprFlowSpecNames;
	private Set<String> allFlowSpecNames;
	
	public FlowSpecExprMap() {
		exprsPerPt = new HashMap<ConnectionPt.InterfacePt, FlowSpecExprs>();
		reprFlowSpecsPerPt = new HashMap<ConnectionPt.InterfacePt, FlowSpecExprs>();
		primPts = new HashSet<ConnectionPt.PrimitivePt>();
		itfPtPerPort = new HashMap<Port, ConnectionPt.InterfacePt>();
		userDefinedFlowSpecNames = new HashSet<String>();
		reprFlowSpecNames = new HashSet<String>();
		allFlowSpecNames = new HashSet<String>();
	}
	
	public void clear() {
		exprsPerPt.clear();
		primPts.clear();
		itfPtPerPort.clear();
	}
	
	public void add(Block block) {
		for (ConnectionPt pt : block.getOwnedPts()) {
			if (pt instanceof ConnectionPt.InterfacePt) {
				exprsPerPt.put((ConnectionPt.InterfacePt)pt, new FlowSpecExprs());
				itfPtPerPort.put(pt.getPort(), (ConnectionPt.InterfacePt)pt);
			} else {
				if (pt instanceof ConnectionPt.PrimitivePt) {
					primPts.add((ConnectionPt.PrimitivePt)pt);
				}
			}
		}
		
		List<AssociationConstraint> constraints = FilterUtils.filter(block.getInheritedConstraints(), AssociationConstraint.class, null);
		
		for (AssociationConstraint c : constraints) {
			for (ConnectionPt pt : block.getOwnedPts()) {
				if (pt instanceof ConnectionPt.InterfacePt) {
					if (pt.field.getName().equals(c.getName())) {
						exprsPerPt.get((ConnectionPt.InterfacePt)pt).addAll(c.getFlowSpecExprs());
					}
				}
			}
		}
	}
	
	public void process() {
		userDefinedFlowSpecNames.clear();
		allFlowSpecNames.clear();
		
		//Extract user-defined names:
		for (FlowSpecExprs exprs : exprsPerPt.values()) {
			for (FlowSpecExpr expr : exprs.getExprs()) {
				userDefinedFlowSpecNames.add(expr.flowSpecName);
				allFlowSpecNames.add(expr.flowSpecName);
			}
		}
		
		//We work with a copy of exprsPerPt, so that
		//this method can be called multiple times without problems:
		Map<ConnectionPt.InterfacePt, FlowSpecExprs> exprsPerPtCopy = new HashMap<ConnectionPt.InterfacePt, FlowSpecExprs>();
		
		for (Map.Entry<ConnectionPt.InterfacePt, FlowSpecExprs> entry : exprsPerPt.entrySet()) {
			exprsPerPtCopy.put(entry.getKey(), new FlowSpecExprs(entry.getValue()));
		}
		
		//Assign initial flow specs to connection points:
		for (Map.Entry<ConnectionPt.InterfacePt, FlowSpecExprs> entry : exprsPerPtCopy.entrySet()) {
			//Make sure every connection point has at least one flow spec:
			if (entry.getValue().isEmpty()) {
				String newExprName = generateFlowSpecName(entry.getKey().field.getName());
				FlowSpecExpr newExpr = FlowSpecExpr.obtain(newExprName);
				entry.getValue().add(newExpr);
			}
			
			//Assign flow specs to overridden connection points, because they must be the same:
			for (ConnectionPt pt : entry.getKey().getOverriddenPts()) {
				exprsPerPtCopy.get((ConnectionPt.InterfacePt)pt).addAll(entry.getValue());
			}
			
			//Assign flow specs to peer ports, because they must be the same:
			if (entry.getKey() instanceof ConnectionPt.InterfacePt) {
				ConnectionPt.InterfacePt sp = (ConnectionPt.InterfacePt)entry.getKey();
				
				for (ConnectionPt.InterfacePt p : sp.getPeerPts()) {
					exprsPerPtCopy.get(itfPtPerPort.get(p.getPort())).addAll(entry.getValue());
				}
			}
		}
		
		//Connected connection points must have the same flow specs or the conjugated flow specs.
		//Repeatedly spread out flow specs until a fixed point is reached: 
		Set<ConnectionPt.InterfacePt> changedPts = new HashSet<ConnectionPt.InterfacePt>(exprsPerPtCopy.keySet());
		Set<ConnectionPt.InterfacePt> newChangedPts = new HashSet<ConnectionPt.InterfacePt>();
		
		while (changedPts.size() > 0) {
			newChangedPts.clear();
			
			for (ConnectionPt.InterfacePt changedPt : changedPts) {
				FlowSpecExprs changedExprs = new FlowSpecExprs();
				changedExprs.addAll(exprsPerPtCopy.get(changedPt));
				
				for (FlowSpecExpr expr : changedExprs.getExprs()) {
					FlowSpecExpr conjExpr = expr.createConjugate();
					
					for (ConnectionPt inwardPt : changedPt.getInwardPts(true)) {
						if (inwardPt instanceof ConnectionPt.InterfacePt) {
							if (exprsPerPtCopy.get((ConnectionPt.InterfacePt)inwardPt).add(expr)) {
								changedPts.add((ConnectionPt.InterfacePt)inwardPt);
							}
						}
					}
					
					for (ConnectionPt inwardPt : changedPt.getInwardPts(false)) {
						if (inwardPt instanceof ConnectionPt.InterfacePt) {
							if (exprsPerPtCopy.get((ConnectionPt.InterfacePt)inwardPt).add(conjExpr)) {
								changedPts.add((ConnectionPt.InterfacePt)inwardPt);
							}
						}
					}
					
					for (ConnectionPt outwardPt : changedPt.getOutwardPts(true)) {
						if (outwardPt instanceof ConnectionPt.InterfacePt) {
							if (exprsPerPtCopy.get((ConnectionPt.InterfacePt)outwardPt).add(expr)) {
								changedPts.add((ConnectionPt.InterfacePt)outwardPt);
							}
						}
					}
					
					for (ConnectionPt outwardPt : changedPt.getInwardPts(false)) {
						if (outwardPt instanceof ConnectionPt.InterfacePt) {
							if (exprsPerPtCopy.get((ConnectionPt.InterfacePt)outwardPt).add(conjExpr)) {
								changedPts.add((ConnectionPt.InterfacePt)outwardPt);
							}
						}
					}
				}
			}
			
			changedPts.clear();
			changedPts.addAll(newChangedPts);
		}
		
		//Determine if there are inconsistencies in interface expressions:
		Map<String, Set<String>> eqInterfaces = new HashMap<String, Set<String>>();
		Map<String, Set<String>> neqInterfaces = new HashMap<String, Set<String>>();
		
		for (String s : allFlowSpecNames) {
			Set<String> eqs = new HashSet<String>();
			Set<String> neqs = new HashSet<String>();
			eqs.add(s);
			
			eqInterfaces.put(s, eqs);
			neqInterfaces.put(s, neqs);
		}
		
		for (FlowSpecExprs exprs : exprsPerPt.values()) {
			for (FlowSpecExpr e1 : exprs.getExprs()) {
				for (FlowSpecExpr e2 : exprs.getExprs()) {
					if (e1.isConjugate == e2.isConjugate) {
						eqInterfaces.get(e1.flowSpecName).add(e2.flowSpecName);
						eqInterfaces.get(e2.flowSpecName).add(e1.flowSpecName);
					} else {
						neqInterfaces.get(e1.flowSpecName).add(e2.flowSpecName);
						neqInterfaces.get(e2.flowSpecName).add(e1.flowSpecName);
					}
				}
			}
		}
		
		boolean done = false;
		
		while (!done) {
			done = true;
			
			for (Set<String> neqItfs : neqInterfaces.values()) {
				for (String neqItf : new HashSet<String>(neqItfs)) {
					if (neqItfs.addAll(eqInterfaces.get(neqItf))) {
						done = false;
					}
				}
			}
		}
		
		for (String s : userDefinedFlowSpecNames) {
			if (neqInterfaces.get(s).contains(s)) {
				String msg = "Interface \"" + s + "\" cannot be equivalent to one or more of the interfaces";
				
				for (String neqItf : neqInterfaces.get(s)) {
					msg += "\n\t" + neqItf;
				}
				
				msg += "\n!";
				throw new Error(msg);
			}
		}
		
		//Pick representative interface expressions for all connection points,
		//preferring user-defined interfaces over generated ones:
		reprFlowSpecsPerPt.clear();
		reprFlowSpecNames.clear();
		reprFlowSpecNames.addAll(userDefinedFlowSpecNames);
		
		for (Map.Entry<ConnectionPt.InterfacePt, FlowSpecExprs> entry : exprsPerPtCopy.entrySet()) {
			FlowSpecExprs reprExprs = new FlowSpecExprs();
			FlowSpecExpr fallback = null;
			
			for (FlowSpecExpr expr : entry.getValue().getExprs()) {
				if (reprFlowSpecNames.contains(expr.flowSpecName)) {
					reprExprs.add(expr);
				} else {
					fallback = expr;
				}
			}
			
			if (reprExprs.isEmpty()) {
				reprExprs.add(fallback);
				reprFlowSpecNames.add(fallback.flowSpecName);
			}
			
			reprFlowSpecsPerPt.put(entry.getKey(), reprExprs);
		}
	}
	
	public Set<String> getReprFlowSpecNames() {
		return Collections.unmodifiableSet(reprFlowSpecNames);
	}
	
	public Set<ConnectionPt.PrimitivePt> getPrimPts() {
		return Collections.unmodifiableSet(primPts);
	}
	
	public Set<ConnectionPt.InterfacePt> getConnectionPts() {
		return Collections.unmodifiableSet(reprFlowSpecsPerPt.keySet());
	}
	
	public Set<FlowSpecExpr> getReprFlowSpecs(ConnectionPt.InterfacePt pt) {
		return reprFlowSpecsPerPt.get(pt).getExprs();
	}
	
	private String generateFlowSpecName(String substr) {
		String attempt = substr + "_FS";
		int index = -1;
		
		while (!allFlowSpecNames.add(attempt)) {
			index++;
			attempt = substr + "_FS" + index;
		}
		
		return attempt;
	}
}
