package lib.blocks.common;

import java.lang.reflect.*;
import java.util.*;

import lib.asal.ASALDataType;
import lib.blocks.ibd1.*;
import lib.blocks.ibd2.*;
import lib.blocks.models.*;
import lib.utils.*;

public abstract class ConnectionPt implements Textifiable {
	private Set<ConnectionPt> connectedPts;
	
	public final Block ownerBlock;
	public final Class<?> diagramClz;
	public final Field field;
	
	private ConnectionPt(Block ownerBlock, Class<?> diagramClz, Field field) {
		this.ownerBlock = ownerBlock;
		this.diagramClz = diagramClz;
		this.field = field;
		
		connectedPts = new HashSet<ConnectionPt>();
	}
	
	/**
	 * Yields the original port object to which this connection point corresponds (may not be unique, especially not between different IBDs).
	 */
	public abstract Port getPort();
	
	/**
	 * Obtains all connection points from super-blocks that are overridden by this connection point. Also includes this connection point.
	 */
	public abstract Set<? extends ConnectionPt> getOverriddenPts();
	
	/**
	 * Obtains all connection points that are connected to this connection point (via a flow).
	 */
	public final Set<ConnectionPt> getConnectedPts() {
		return Collections.unmodifiableSet(connectedPts);
	}
	
	@Override
	public String toString() {
		String result = field.getName();
		
		if (InterfacePt.class.isAssignableFrom(getClass())) {
			int portIndex = ((InterfacePt)this).getPortId().index;
			
			if (portIndex >= 0) {
				result += "[" + portIndex + "]";
			}
		}
		
		return "Port \"" + result + "\" of block \"" + ownerBlock.id + "\" (" + diagramClz.getCanonicalName() + ")";
	}
	
	public abstract Set<ConnectionPt> getInwardPts(boolean mustPropagate);
	
	public abstract Set<ConnectionPt> getOutwardPts(boolean mustPropagate);
	
	protected abstract void checkConsistency() throws ConnectionPtException;
	
	private static <T extends ConnectionPt> void removeClz(Set<T> pts, Class<?> clz) {
		pts.removeIf((pt) -> {
			return clz.isAssignableFrom(pt.getClass());
		});
	}
	
	private static <T extends ConnectionPt> void retainClz(Set<T> pts, Class<?> clz) {
		pts.removeIf((pt) -> {
			return !clz.isAssignableFrom(pt.getClass());
		});
	}
	
	/**
	 * Point that is owned by a subcomponent of an IBD2 (or `member').
	 */
	public interface InternalPt {
		/**
		 * Connection point that corresponds with this object.
		 */
		public ConnectionPt asPt();
		
		/**
		 * Id of the subcomponent of the IBD2 that owns the port that corresponds with this point.
		 */
		public NameIndexId getMemberId();
	}
	
	/**
	 * Primitive point (always owned by an IBD2 member).
	 */
	public abstract static class PrimitivePt extends ConnectionPt implements InternalPt {
		private NameIndexId memberId;
		
		private PrimitivePt(Block ownerBlock, Class<?> diagramClz, Field field, NameIndexId memberId) {
			super(ownerBlock, diagramClz, field);
			
			this.memberId = memberId;
		}
		
		@Override
		public ConnectionPt asPt() {
			return this;
		}
		
		@Override
		public NameIndexId getMemberId() {
			return memberId;
		}
		
		public abstract Dir getDir();
		
		public abstract ASALDataType getType();
		
		public Set<String> getConnectedInterfaceNames() {
			Set<String> result = new HashSet<String>();
			
			for (ConnectionPt pt : getConnectedPts()) {
				if (pt instanceof InterfacePt) {
					result.add(pt.field.getName());
				}
			}
			
			return Collections.unmodifiableSet(result);
		}
		
		@Override
		public final Set<ConnectionPt> getInwardPts(boolean mustPropagate) {
			return Collections.emptySet();
		}
		
		@Override
		public final Set<ConnectionPt> getOutwardPts(boolean mustPropagate) {
			return getConnectedPts();
		}
		
		@Override
		public final Set<PrimitivePt> getOverriddenPts() {
			Set<PrimitivePt> result = new HashSet<PrimitivePt>();
			
			for (Block superBlock : ownerBlock.getLineage()) {
				for (ConnectionPt pt : superBlock.getOwnedPts()) {
					if (pt instanceof PrimitivePt) {
						if (pt.field.getName().equals(field.getName())) {
							result.add((PrimitivePt)pt);
						}
					}
				}
			}
			
			return result;
		}
		
		@Override
		protected void checkConsistency() throws ConnectionPtException {
			// Do nothing.
		}
	}
	
	/**
	 * Primitive input port.
	 */
	public final static class InPortPt extends PrimitivePt {
		private ASALDataType type;
		
		public final InPort<?> port;
		
		public InPortPt(Block ownerBlock, Class<?> diagramClz, Field f, NameIndexId memberId, InPort<?> port) throws ReflectionException {
			super(ownerBlock, diagramClz, f, memberId);
			
			this.port = port;
			
			ParameterizedField pf = ParameterizedField.obtain(f, InPort.class);
			type = ASALDataType.get(pf.parameterType);
		}
		
		@Override
		public Port getPort() {
			return port;
		}
		
		@Override
		public Dir getDir() {
			return Dir.IN;
		}
		
		@Override
		public ASALDataType getType() {
			return type;
		}
		
		@Override
		public String textify(LOD lod) {
			String result = lod.abbreviate(field.getName());
			
			if (lod.includesType) {
				result += ": " + type.javaClass.getSimpleName();
			}
			
			if (lod.includesDir) {
				result = "IN " + result;
			}
			
			return result;
		}
	}
	
	/**
	 * Primitive output port.
	 */
	public final static class OutPortPt extends PrimitivePt {
		private ASALDataType type;
		
		public final OutPort<?> port;
		
		public OutPortPt(Block ownerBlock, Class<?> diagramClz, Field f, NameIndexId memberId, OutPort<?> port) throws ReflectionException {
			super(ownerBlock, diagramClz, f, memberId);
			
			this.port = port;
			
			ParameterizedField pf = ParameterizedField.obtain(f, OutPort.class);
			type = ASALDataType.get(pf.parameterType);
		}
		
		@Override
		public Port getPort() {
			return port;
		}
		
		@Override
		public Dir getDir() {
			return Dir.OUT;
		}
		
		@Override
		public ASALDataType getType() {
			return type;
		}
		
		@Override
		public String textify(LOD lod) {
			String result = lod.abbreviate(field.getName());
			
			if (lod.includesType) {
				result += ": " + type.javaClass.getSimpleName();
			}
			
			if (lod.includesDir) {
				result = "OUT " + result;
			}
			
			return result;
		}
	}
	
	/**
	 * Point that corresponds with an interface port (rather than a primitive port).
	 * Can belong to an IBD2 member or to the IBD2 itself.
	 */
	public static abstract class InterfacePt extends ConnectionPt {
		private final InterfacePort port;
		private final NameIndexId portId;
		
		protected InterfacePt(Block ownerBlock, Class<?> diagramClz, Field f, InterfacePort port, int portIndex) {
			super(ownerBlock, diagramClz, f);
			
			this.port = port;
			
			portId = new NameIndexId(f.getName(), portIndex);
		}
		
		@Override
		public InterfacePort getPort() {
			return port;
		}
		
		public NameIndexId getPortId() {
			return portId;
		}
		
		public FlowSpecExpr getFlowSpecExpr() {
			return port.flowSpecExpr;
		}
		
		@Override
		public final Set<InterfacePt> getOverriddenPts() {
			Set<InterfacePt> result = new HashSet<InterfacePt>();
			
			for (Block superBlock : ownerBlock.getLineage()) {
				for (ConnectionPt pt : superBlock.getOwnedPts()) {
					if (pt instanceof InterfacePt) {
						if (pt.field.getName().equals(field.getName())) {
							result.add((InterfacePt)pt);
						}
					}
				}
			}
			
			return result;
		}
		
		/**
		 * Includes self and inherited interface points.
		 */
		public Set<InterfacePt> getPeerPts() {
			Set<InterfacePt> result = new HashSet<InterfacePt>();
			
			for (Block superBlock : ownerBlock.getLineage()) {
				for (ConnectionPt pt : superBlock.getOwnedPts()) {
					if (pt.field.getName().equals(field.getName())) {
						if (pt instanceof InterfacePt) {
							result.add((InterfacePt)pt);
						}
					}
				}
			}
			
			return result;
		}
	}
	
	/**
	 * Interface point that belongs to an IBD2 member.
	 */
	public final static class ExposedPt extends InterfacePt implements InternalPt {
		private NameIndexId memberId;
		
		public ExposedPt(Block ownerBlock, Class<?> diagramClz, Field f, NameIndexId memberId, InterfacePort port, int portIndex) {
			super(ownerBlock, diagramClz, f, port, portIndex);
			
			this.memberId = memberId;
		}
		
		@Override
		public ConnectionPt asPt() {
			return this;
		}
		
		@Override
		public NameIndexId getMemberId() {
			return memberId;
		}
		
		//TODO does this work?
		public ExposedPt getOtherSide() {
			Iterator<ConnectionPt> q = getConnectedPts().iterator();
			return q.hasNext() ? (ExposedPt) q.next() : null;
		}
		
		/**
		 * Obtains all connection points that (i) are defined by the same block (or one of its super-blocks), and (ii) are connected to components of the block (and not to its environment).
		 */
		public Set<ExposingPt> getReflectionPts() {
			return FilterUtils.filter(getOverriddenPts(), ExposingPt.class, null);
		}
		
		@Override
		public Set<ConnectionPt> getInwardPts(boolean mustPropagate) {
			Set<ConnectionPt> result = new HashSet<ConnectionPt>();
			
			for (ExposingPt pt : getReflectionPts()) {
				result.addAll(pt.getInwardPts(mustPropagate));
			}
			
			return Collections.unmodifiableSet(result);
		}
		
		@Override
		public Set<ConnectionPt> getOutwardPts(boolean mustPropagate) {
			Set<ConnectionPt> result = new HashSet<ConnectionPt>(getConnectedPts());
			
			if (mustPropagate) {
				// We just exited a subcomponent;
				// now we want to exit the component that contains that subcomponent:
				retainClz(result, ExposingPt.class);
			} else {
				// We just exited a subcomponent;
				// now we want another exposed pt that mirrors this one:
				retainClz(result, ExposedPt.class);
			}
			
			return Collections.unmodifiableSet(result);
		}
		
		@Override
		protected void checkConsistency() throws ConnectionPtException {
			if (getPort().getConnectedPorts().size() > 0) {
				for (Port p : getPort().getConnectedPorts()) {
					if (!(p instanceof InterfacePort)) {
						throw new ConnectionPtException(this, "Cannot connect these ports!");
					}
				}
				
				if (getPort().getConnectedPorts().size() > 1) {
					throw new ConnectionPtException(this, "Too many connections for interface flow port " + getPort() + "!");
				}
			}
		}
		
		@Override
		public String textify(LOD lod) {
			return field.getName();
		}
	}
	
	/**
	 * Point that belongs to the IBD2 itself.
	 */
	public final static class ExposingPt extends InterfacePt {
		public ExposingPt(Block ownerBlock, Class<?> diagramClz, Field f, InterfacePort port, int portIndex) {
			super(ownerBlock, diagramClz, f, port, portIndex);
		}
		
		/**
		 * Obtains all connection points that (i) are defined by the same block (or one of its super-blocks), and (ii) are connected to environment of the block (and not to its components).
		 */
		public Set<ExposedPt> getReflectionPts() {
			return FilterUtils.filter(getOverriddenPts(), ExposedPt.class, null);
		}
		
		@Override
		public Set<ConnectionPt> getInwardPts(boolean mustPropagate) {
			Set<ConnectionPt> result = new HashSet<ConnectionPt>(getConnectedPts());
			
			if (mustPropagate) {
				// We just entered a component;
				// now we want to enter a subcomponent or an input/output port:
				removeClz(result, ExposingPt.class);
			} else {
				// We just entered a component;
				// now we want to exit that component again:
				retainClz(result, ExposingPt.class);
			}
			
			return Collections.unmodifiableSet(result);
		}
		
		@Override
		public Set<ConnectionPt> getOutwardPts(boolean mustPropagate) {
			Set<ConnectionPt> result = new HashSet<ConnectionPt>();
			
			for (ExposedPt pt : getReflectionPts()) {
				result.addAll(pt.getOutwardPts(mustPropagate));
			}
			
			return Collections.unmodifiableSet(result);
		}
		
		@Override
		protected void checkConsistency() throws ConnectionPtException {
			for (Port p : getPort().getConnectedPorts()) {
				if (p instanceof InterfacePort) {
					throw new ConnectionPtException(this, "Cannot connect these ports!");
				}
			}
		}
		
		@Override
		public String textify(LOD lod) {
			return field.getName();
		}
	}
	
	/**
	 * Figures out which connection points are connected based on whether the ports with which they correspond are connected; then connects the connection points accordingly.
	 */
	public static void populateAll(IBD2Target instance, Collection<ConnectionPt> pts) throws ConnectionPtException {
		Map<Port, ConnectionPt> ptPerPort = new HashMap<Port, ConnectionPt>();
		
		for (ConnectionPt pt : pts) {
			pt.checkConsistency();
			
			if (ptPerPort.containsKey(pt.getPort())) {
				throw new Error("Double binding for " + pt.getPort() + "!");
			}
			
			ptPerPort.put(pt.getPort(), pt);
			pt.connectedPts.clear();
		}
		
		for (ConnectionPt pt1 : pts) {
			for (Port p2 : pt1.getPort().getConnectedPorts()) {
				connect(ptPerPort, pt1, p2);
			}
		}
	}
	
	private static void connect(Map<Port, ConnectionPt> ptPerPort, ConnectionPt pt1, Port p2) {
		ConnectionPt pt2 = ptPerPort.get(p2);
		
		if (pt2 == null) {
			String msg = pt1.toString() + " connects to a port that is not accessible from the current IBD! Accessible ports:";
			
			for (Map.Entry<Port, ConnectionPt> entry : ptPerPort.entrySet()) {
				msg += "\n\t" + entry.getValue().toString();
			}
			
			throw new Error(msg);
		}
		
		if (pt1 instanceof ExposedPt || pt2 instanceof ExposedPt) {
			if (pt1 instanceof ExposedPt && pt2 instanceof ExposedPt) {
				pt2.connectedPts.add(pt1);
				pt1.connectedPts.add(pt2);
			} else {
				throw new Error("Illegal source/target combination for flow:\n\t" + pt1.toString() + "\n\t" + pt2.toString());
			}
		} else {
			if (pt1 instanceof PrimitivePt || pt2 instanceof PrimitivePt) {
				pt2.connectedPts.add(pt1);
				pt1.connectedPts.add(pt2);
			} else {
				throw new Error("Illegal source/target combination for flow:\n\t" + pt1.toString() + "\n\t" + pt2.toString());
			}
		}
	}
}
