package lib.blocks.common;

import java.lang.reflect.*;
import java.util.*;

import lib.blocks.bdds.Assoc;
import lib.blocks.models.*;
import lib.utils.ParameterizedField;

/**
 * Used to store information about which Assocs are connected.
 */
public class BidirMap {
	public static class Bidir {
		private Assoc<?> assoc1;
		private Assoc<?> assoc2;
		private Field field1;
		private Field field2;
		private Block owningBlock1;
		private Block owningBlock2;
		private Block assocBlock;
		private int minCount1;
		private int maxCount1;
		private int minCount2;
		private int maxCount2;
		
		private Bidir(ModelLib lib, ParameterizedField pf1, ParameterizedField pf2, Assoc<?> assoc1, Assoc<?> assoc2) {
			this.assoc1 = assoc1;
			this.assoc2 = assoc2;
			
			field1 = pf1.field;
			field2 = pf2.field;
			owningBlock1 = lib.get(pf2.parameterType);
			owningBlock2 = lib.get(pf1.parameterType);
			assocBlock = assoc1.assocBlock != null ? lib.get(assoc1.assocBlock) : null;
			minCount1 = assoc1.minCount;
			maxCount1 = assoc1.maxCount;
			minCount2 = assoc2.minCount;
			maxCount2 = assoc2.maxCount;
		}
		
		private Bidir(Bidir source) {
			field1 = source.field1;
			field2 = source.field2;
		}
		
		public Field getField1() {
			return field1;
		}
		
		public Field getField2() {
			return field2;
		}
		
		public Block getOwningBlock1() {
			return owningBlock1;
		}
		
		public Block getOwningBlock2() {
			return owningBlock2;
		}
		
		public Block getAssocBlock() {
			return assocBlock;
		}
		
		public int getMinCount1() {
			return minCount1;
		}
		
		public int getMaxCount1() {
			return maxCount1;
		}
		
		public int getMinCount2() {
			return minCount2;
		}
		
		public int getMaxCount2() {
			return maxCount2;
		}
		
		public boolean isSame(Bidir other) {
			if (field1 != other.field1 || field2 != other.field2) {
				return false;
			}
			
			if (assoc1 != other.assoc1 || assoc2 != other.assoc2) {
				return false;
			}
			
			return true;
		}
	}
	
	private Map<String, Bidir> bidirPerFieldName;
	private Map<Assoc<?>, ParameterizedField> fieldPerAssoc;
	private Class<?> targetClz;
	private ModelLib lib;
	
	public BidirMap(ModelLib lib, Class<?> targetClz, Map<Assoc<?>, ParameterizedField> fieldPerAssoc) {
		this.fieldPerAssoc = fieldPerAssoc;
		this.targetClz = targetClz;
		this.lib = lib;
		
		bidirPerFieldName = new HashMap<String, Bidir>();
	}
	
	public Bidir get(String fieldName1) {
		return bidirPerFieldName.get(fieldName1);
	}
	
	public void add(Assoc<?> assoc1, Assoc<?> assoc2) {
		if (assoc1 == assoc2) {
			throw new Error("Cannot make association bidirectional with itself!");
		}
		
		ParameterizedField pf1 = fieldPerAssoc.get(assoc1);
		ParameterizedField pf2 = fieldPerAssoc.get(assoc2);
		
		if (pf1 == null || pf2 == null) {
			throw new Error("Cannot reference associations outside of this BDD!");
		}
		
		if (pf1.parameterType != pf2.field.getDeclaringClass()) {
			throw new Error("Association should reference other association's declaring type, but references " + pf1.parameterType.getCanonicalName() + "!");
		}
		
		if (pf2.parameterType != pf1.field.getDeclaringClass()) {
			throw new Error("Association should reference other association's declaring type, but references " + pf2.parameterType.getCanonicalName() + "!");
		}
		
		if (pf1.field.getDeclaringClass().isAssignableFrom(pf2.field.getDeclaringClass())) {
			throw new Error("Cannot define bidirectional associations between a class and an association class!");
		}
		
		if (pf2.field.getDeclaringClass().isAssignableFrom(pf1.field.getDeclaringClass())) {
			throw new Error("Cannot define bidirectional associations between a class and an association class!");
		}
		
		if (pf1.field.getDeclaringClass() == targetClz) {
			add(new Bidir(lib, pf1, pf2, assoc1, assoc2));
		}
		
		if (pf2.field.getDeclaringClass() == targetClz) {
			add(new Bidir(lib, pf2, pf1, assoc2, assoc1));
		}
	}
	
	private void add(Bidir bidir) {
		Bidir existing = bidirPerFieldName.get(bidir.field1.getName());
		
		if (existing != null) {
			if (!existing.isSame(bidir)) {
				throw new Error("Cannot make association bidirectional with multiple other associations!");
			}
		} else {
			bidirPerFieldName.put(bidir.field1.getName(), bidir);
		}
	}
}
