package lib.behave.proto;

import java.util.*;
import java.util.function.Predicate;

import lib.asal.ASALContextDecls;
import lib.asal.parsing.ASALException;
import lib.asal.parsing.api.ASALStatement;
import lib.behave.*;

public class DeuteroVertex {
	private Class<?> clz;
	private DeuteroVertex parentVertex;
	private List<DeuteroVertex> childVertices;
	private List<DeuteroVertex> initialVertices;
	
	private DeuteroTransition onEntry;
	private List<DeuteroTransition> onDo;
	private DeuteroTransition onExit;
	
	public DeuteroVertex(ASALContextDecls context, ProtoVertex vertex) throws ASALException {
		this.clz = vertex.clz;
		
		if (vertex.onEntry != null) {
			onEntry = vertex.onEntry.parse(context, false);
		}
		
		onDo = new ArrayList<DeuteroTransition>();
		
		for (ProtoTransition transition : vertex.onDo) {
			onDo.add(transition.parse(context, false));
		}
		
		if (vertex.onExit != null) {
			onExit = vertex.onExit.parse(context, false);
		}
		
		childVertices = new ArrayList<DeuteroVertex>();
		initialVertices = new ArrayList<DeuteroVertex>();
	}
	
	public Class<?> getClz() {
		return clz;
	}
	
	public DeuteroVertex getParentVertex() {
		return parentVertex;
	}
	
	public void setParentVertex(DeuteroVertex parentVertex) {
		this.parentVertex = parentVertex;
	}
	
	public List<DeuteroVertex> getChildVertices() {
		return childVertices;
	}
	
	public List<DeuteroVertex> getInitialVertices() {
		return initialVertices;
	}
	
	public DeuteroTransition getOnEntry() {
		return onEntry;
	}
	
	public List<DeuteroTransition> getOnDo() {
		return onDo;
	}
	
	public DeuteroTransition getOnExit() {
		return onExit;
	}
	
	/**
	 * Constructs a list with all ancestors of this vertex, including itself.
	 * The left-most element in the list will be the root container (which must be a state machine), and
	 * the right-most element in the list will be this vertex.
	 */
	public List<DeuteroVertex> getLineage() {
		List<DeuteroVertex> result;
		
		if (parentVertex != null) {
			result = parentVertex.getLineage();
		} else {
			result = new ArrayList<DeuteroVertex>();
		}
		
		result.add(this);
		return result;
	}
	
	public Map<DeuteroVertex, List<ASALStatement>> getExitBehaviourPerChildlessDescendantVertex(Predicate<DeuteroVertex> filter) {
		Map<DeuteroVertex, List<ASALStatement>> result = new HashMap<DeuteroVertex, List<ASALStatement>>();
		addExitBehaviourPerChildlessDescendantVertex(result, new ArrayList<ASALStatement>(), filter);
		return result;
	}
	
	private void addExitBehaviourPerChildlessDescendantVertex(Map<DeuteroVertex, List<ASALStatement>> destination, List<ASALStatement> exitBehaviourSoFar, Predicate<DeuteroVertex> filter) {
		List<ASALStatement> exitBehaviour = new ArrayList<ASALStatement>(exitBehaviourSoFar);
		
		if (onExit != null) {
			exitBehaviour.add(0, onExit.getStatement());
		}
		
		if (childVertices.size() > 0) {
			for (DeuteroVertex childVertex : childVertices) {
				childVertex.addExitBehaviourPerChildlessDescendantVertex(destination, exitBehaviour, filter);
			}
		} else {
			if (filter.test(this)) {
				destination.put(this, exitBehaviour);
			}
		}
	}
	
	public Map<DeuteroVertex, List<ASALStatement>> getExitBehaviourPerFinalVertex() {
		Map<DeuteroVertex, List<ASALStatement>> result = new HashMap<DeuteroVertex, List<ASALStatement>>();
		List<ASALStatement> exitBehaviour = new ArrayList<ASALStatement>();
		
		if (onExit != null) {
			exitBehaviour.add(0, onExit.getStatement());
		}
		
		if (childVertices.size() > 0) {
			for (DeuteroVertex childVertex : childVertices) {
				if (FinalVertex.class.isAssignableFrom(childVertex.clz)) {
					result.put(childVertex, exitBehaviour); //There should be only one!
				}
			}
		} else {
			result.put(this, exitBehaviour);
		}
		
		return result;
	}
	
	public Set<DeuteroVertex> getChildlessDescendantVertices(Predicate<DeuteroVertex> filter) {
		Set<DeuteroVertex> result = new HashSet<DeuteroVertex>();
		addChildlessDescendantVertices(filter, result);
		return result;
	}
	
	private void addChildlessDescendantVertices(Predicate<DeuteroVertex> filter, Set<DeuteroVertex> destination) {
		if (childVertices.size() > 0) {
			for (DeuteroVertex childVertex : childVertices) {
				childVertex.addChildlessDescendantVertices(filter, destination);
			}
		} else {
			if (filter.test(this)) {
				destination.add(this);
			}
		}
	}
}
