package lib.behave.proto;

import java.util.*;

import lib.asal.*;
import lib.asal.parsing.ASALException;
import lib.behave.*;

/**
 * In this machine, ASAL has been parsed and validated.
 */
public class DeuteroStateMachine extends ASALContext {
	public final Class<? extends StateMachine> clz;
	public final DeuteroVertex rootVertex;
	public final Set<DeuteroVertex> vertices;
	public final Set<DeuteroVertex> initialVertices;
	public final Set<DeuteroTransition> transitions;
	
	public final ProtoStateMachine legacy;
	
	public DeuteroStateMachine(ProtoStateMachine source) throws ASALException {
		clz = source.clz;
		legacy = source;
		
		//Variables:
		for (Map.Entry<String, Variable> entry : source.inPortVars.entrySet()) {
			inPortVars.put(entry.getKey(), new ASALVariableImpl<Object>(entry.getKey(), entry.getValue(), ASALVarOrigin.STM_PORT));
		}
		
		for (Map.Entry<String, Variable> entry : source.outPortVars.entrySet()) {
			outPortVars.put(entry.getKey(), new ASALVariableImpl<Object>(entry.getKey(), entry.getValue(), ASALVarOrigin.STM_PORT));
		}
		
		for (Map.Entry<String, Variable> entry : source.stateMachineVars.entrySet()) {
			stateMachineVars.put(entry.getKey(), new ASALVariableImpl<Object>(entry.getKey(), entry.getValue(), ASALVarOrigin.STM_VAR));
		}
		
		//Functions:
		for (Map.Entry<String, Function> entry : source.functions.entrySet()) {
			ASALFunction fct = new ASALFunctionImpl(entry.getKey(), entry.getValue());
			functions.put(fct.getName(), fct);
			
			try {
				fct.parse(source);
			} catch (ASALException e) {
				throw new ASALException("State machine: " + source.clz.getCanonicalName() + "\nFunction: " + fct.getName() + "()", e);
			}
		}
		
		//First copy vertices blindly:
		vertices = new HashSet<DeuteroVertex>();
		Map<ProtoVertex, DeuteroVertex> newVertexPerOldVertex = new HashMap<ProtoVertex, DeuteroVertex>();
		
		for (ProtoVertex v : source.vertices.values()) {
			DeuteroVertex w = new DeuteroVertex(source, v);
			newVertexPerOldVertex.put(v, w);
			vertices.add(w);
		}
		
		//Resolve references to root/initial/parent/child vertices:
		rootVertex = newVertexPerOldVertex.get(source.rootVertex);
		initialVertices = new HashSet<DeuteroVertex>();
		
		for (ProtoVertex initialVertex : source.initialVertices) {
			initialVertices.add(newVertexPerOldVertex.get(initialVertex));
		}
		
		for (Map.Entry<ProtoVertex, DeuteroVertex> entry : newVertexPerOldVertex.entrySet()) {
			for (ProtoVertex initialVertex : entry.getKey().initialVertices) {
				entry.getValue().getInitialVertices().add(newVertexPerOldVertex.get(initialVertex));
			}
			
			if (entry.getKey().parentVertex != null) {
				entry.getValue().setParentVertex(newVertexPerOldVertex.get(entry.getKey().parentVertex));
			}
			
			for (ProtoVertex childVertex : entry.getKey().childVertices) {
				entry.getValue().getChildVertices().add(newVertexPerOldVertex.get(childVertex));
			}
		}
		
		//Copy transitions:
		transitions = new HashSet<DeuteroTransition>();
		
		for (ProtoTransition transition : source.transitions) {
			try {
				//Parse the ASAL code:
				boolean isInitCode = source.initialVertices.contains(transition.sourceState);
				DeuteroTransition newTransition = transition.parse(source, isInitCode);
				newTransition.setSourceVertex(newVertexPerOldVertex.get(transition.sourceState));
				newTransition.setTargetVertex(newVertexPerOldVertex.get(transition.targetState));
				newTransition.setIsLocal(transition.isLocal);
				transitions.add(newTransition);
			} catch (ASALException e) {
				throw new ASALException("State machine: " + source.clz.getCanonicalName() + "\nTransition: " + transition, e);
			}
		}
	}
}
