package lib.asal.parsing.api;

import java.util.*;

import lib.asal.*;
import lib.asal.parsing.*;
import lib.behave.Function;
import lib.behave.Variable;
import lib.utils.LOD;

public class ASALFunctionCall extends ASALExpr {
	private String fctName;
	private ASALSeqExpr params;
	
	public ASALFunctionCall(ASALSyntaxTreeAPI parent, ASALSyntaxTree tree) {
		super(parent, tree);
		
		fctName = tree.getPropery("fct");
		params = createAPI("params", ASALSeqExpr.class, true);
	}
	
	public String getFctName() {
		return fctName;
	}
	
	public List<ASALExpr> getParams() {
		if (params != null) {
			return Collections.unmodifiableList(params.getList());
		}
		
		return Collections.emptyList();
	}
	
	public Function resolveFunctionDecl(ASALContextDecls context) throws ASALException {
		Function result = context.getFunctionDecl(fctName);
		
		if (result == null) {
			throw new ASALException("Unknown function", this, context.getScopeSuggestions());
		}
		
		return result;
	}
	
	public ASALFunction resolveFunction(ASALContext context) {
		return context.getFunction(fctName);
	}
	
	@Override
	public ASALDataType getType(ASALContextDecls context) throws ASALException {
		return resolveFunctionDecl(context).getReturnType();
	}
	
	@Override
	public ASALDataType getType(ASALContext context) throws ASALException {
		return resolveFunction(context).getReturnType();
	}

	@Override
	public void validate(ASALContextDecls context) throws ASALException {
		Function f = resolveFunctionDecl(context);
		
		List<ASALExpr> foundParams = getParams();
		List<String> expectedParamNames = f.getParamNames();
		List<Variable> expectedParamDecls = f.getParamDecls();
		
		if (foundParams.size() != expectedParamDecls.size()) {
			throw new ASALException("Expected " + expectedParamDecls.size() + " parameters but found " + foundParams.size(), this);
		}
		
		for (int index = 0; index < foundParams.size(); index++) {
			ASALDataType foundParamType = foundParams.get(index).getType(context);
			ASALDataType expectedParamType = expectedParamDecls.get(index).getType();
			
			if (foundParamType != expectedParamType) {
				throw new ASALException("Expected " + expectedParamType.name + " as type of parameter " + (index + 1) + " (\"" + expectedParamNames.get(index) + "\") but found " + foundParamType.name, this);
			}
		}
	}
	
	@Override
	public String textify(LOD lod) {
		if (params != null) {
			return fctName + "(" + params.textify(lod) + ")";
		}
		
		return fctName + "()";
	}
}
