package lib.asal.parsing.api;

import java.util.List;

import lib.asal.*;
import lib.asal.parsing.*;
import lib.utils.LOD;

public class ASALBinaryExpr extends ASALExpr {
	private ASALExpr lhs;
	private ASALExpr rhs;
	private String op;
	
	public ASALBinaryExpr(ASALSyntaxTreeAPI parent, ASALSyntaxTree tree) {
		super(parent, tree);
		
		lhs = createAPI("lhs", ASALExpr.class, false);
		rhs = createAPI("rhs", ASALExpr.class, false);
		op = tree.get("op").getFirstToken().text;
	}
	
	public ASALBinaryExpr(ASALExpr lhs, ASALExpr rhs, String op) {
		super(lhs.getParent(), lhs.getTree());
		
		this.lhs = lhs.getTree().createAPI(this, ASALExpr.class);
		this.lhs = rhs.getTree().createAPI(this, ASALExpr.class);
		this.op = op;
	}
	
	public ASALExpr getLhs() {
		return lhs;
	}
	
	public ASALExpr getRhs() {
		return rhs;
	}
	
	public String getOp() {
		return op;
	}
	
	@Override
	public ASALDataType getType(ASALContextDecls context) throws ASALException {
		switch (op) {
			case "+":
				if (lhs.getType(context) == ASALDataType.STRING) {
					return ASALDataType.STRING;
				}
				
				if (rhs.getType(context) == ASALDataType.STRING) {
					return ASALDataType.STRING;
				}
				
				return ASALDataType.NUMBER;
			case "-":
			case "*":
			case "/":
			case "%":
				return ASALDataType.NUMBER;
			case "and":
			case "or":
			case "=":
			case "<>":
			case ">=":
			case "<=":
			case ">":
			case "<":
				return ASALDataType.BOOLEAN;
		}
		
		throw new Error("Unknown operation (\"" + op + "\")!");
	}
	
	@Override
	public ASALDataType getType(ASALContext context) throws ASALException {
		switch (op) {
			case "+":
				if (lhs.getType(context) == ASALDataType.STRING) {
					return ASALDataType.STRING;
				}
				
				if (rhs.getType(context) == ASALDataType.STRING) {
					return ASALDataType.STRING;
				}
				
				return ASALDataType.NUMBER;
			case "-":
			case "*":
			case "/":
			case "%":
				return ASALDataType.NUMBER;
			case "and":
			case "or":
			case "=":
			case "<>":
			case ">=":
			case "<=":
			case ">":
			case "<":
				return ASALDataType.BOOLEAN;
		}
		
		throw new Error("Unknown operation (\"" + op + "\")!");
	}
	
	@Override
	public void validate(ASALContextDecls context) throws ASALException {
		lhs.validate(context);
		rhs.validate(context);
		
		switch (op) {
			case "+":
				if (lhs.getType(context) == ASALDataType.STRING) {
					return;
				}
				
				if (rhs.getType(context) == ASALDataType.STRING) {
					return;
				}
				
				if (lhs.getType(context) != ASALDataType.NUMBER) {
					throw new ASALException("Invalid operand type", lhs);
				}
				
				if (rhs.getType(context) != ASALDataType.NUMBER) {
					throw new ASALException("Invalid operand type", rhs);
				}
				
				return;
			case "-":
			case "*":
			case "/":
			case "%":
				if (lhs.getType(context) != ASALDataType.NUMBER) {
					throw new ASALException("Invalid operand type", lhs);
				}
				
				if (rhs.getType(context) != ASALDataType.NUMBER) {
					throw new ASALException("Invalid operand type", rhs);
				}
				
				return;
			case "and":
			case "or":
				if (!lhs.getType(context).isBooleanType) {
					throw new ASALException("Invalid operand type", lhs);
				}
				
				if (!rhs.getType(context).isBooleanType) {
					throw new ASALException("Invalid operand type", rhs);
				}
				
				return;
			case "=":
			case "<>":
				if (!lhs.getType(context).isEquatable(rhs.getType(context))) {
					throw new ASALException("Data type mismatch", rhs);
				}
				
				return;
			case ">=":
			case "<=":
			case ">":
			case "<":
				if (!lhs.getType(context).isOrderedType) {
					throw new ASALException("Invalid operand type", lhs);
				}
				
				if (!rhs.getType(context).isOrderedType) {
					throw new ASALException("Invalid operand type", rhs);
				}
				
				return;
		}
		
		throw new Error("Should not happen!");
	}
	
	@Override
	public String textify(LOD lod) {
		return lhs.textify(lod) + " " + op + " " + rhs.textify(lod);
	}
	
	public static ASALExpr fromList(List<ASALExpr> exprs, String op) {
		if (exprs.size() > 0) {
			ASALExpr prevExpr = exprs.get(0);
			
			for (int index = 1; index < exprs.size(); index++) {
				prevExpr = new ASALBinaryExpr(prevExpr, exprs.get(index), op);
			}
			
			return prevExpr;
		}
		
		return null;
	}
}
