package lib.asal;

import java.util.*;

import lib.asal.parsing.ASALException;
import lib.asal.parsing.api.*;

public abstract class ASALVisitor<T> extends ASALHandler<T> {
	protected final ASALContext context;
	
	public ASALVisitor(ASALContext context) {
		this.context = context;
	}
	
	public abstract T visitFctDef(ASALFunction fct) throws ASALException;
	
	protected final T visitVar(ASALVariable<?> node) throws ASALException {
		return handle(node);
	}
	
	protected final T visit(ASALAssignStatement node) throws ASALException {
		return handle(node, handle(node.resolveVar(context)), visitExpr(node.getExpression()));
	}
	
	protected final T visit(ASALBinaryExpr node) throws ASALException {
		return handle(node, visitExpr(node.getLhs()), visitExpr(node.getRhs()));
	}
	
	protected final T visit(ASALEmptyStatement node) throws ASALException {
		return handle(node);
	}
	
	protected final T visit(ASALFunctionCall node) throws ASALException {
		T visitedFct = visitFctDef(node.resolveFunction(context));
		Object[] visitedParams = new Object[0];
		return handle(node, visitedFct, visitedParams);
	}
	
	protected final T visit(ASALFunctionCallStatement node) throws ASALException {
		T visitedFct = visitFctDef(node.resolveFunction(context));
		Object[] visitedParams = new Object[0];
		return handle(node, visitedFct, visitedParams);
	}
	
	protected final T visit(ASALIfStatement node) throws ASALException {
		T condition = visitExpr(node.getCondition());
		T thenBranch = visitStat(node.getThenBranch());
		T elseBranch = null;
		
		if (node.getElseBranch() != null) {
			elseBranch = visitStat(node.getElseBranch());
		}
		
		return handle(node, condition, thenBranch, elseBranch);
	}
	
	protected final T visit(ASALWhileStatement node) throws ASALException {
		T condition = visitExpr(node.getCondition());
		T body = visitStat(node.getBody());
		return handle(node, condition, body);
	}
	
	protected final T visit(ASALLiteral node) throws ASALException {
		return handle(node);
	}
	
	protected final T visit(ASALReturnStatement node) throws ASALException {
		return handle(node, visitExpr(node.getExpression()));
	}
	
	protected final T visit(ASALSeqStatement node) throws ASALException {
		return handle(node, visitStat(node.getStatement()), visitStat(node.getSuccessor()));
	}
	
	protected final T visit(ASALUnaryExpr node) throws ASALException {
		return handle(node, visitExpr(node.getExpr()));
	}
	
	protected final T visit(ASALVarRef node) throws ASALException {
		return handle(node, visitVar(node.resolveVar(context)));
	}
	
	protected final T visit(ASALTrigger trigger) throws ASALException {
		return handle(trigger, visitExpr(trigger.getExpr()));
	}
	
	protected final T visit(ASALTimeout timeout) throws ASALException {
		return handle(timeout, visitExpr(timeout.getDuration()));
	}
	
	protected final T visit(ASALCall call) throws ASALException {
		T visitedFct = visitFctDef(call.resolveFunction(context));
		return handle(call, visitedFct);
	}
	
	public final T visitEvent(ASALEvent evt) throws ASALException {
		if (evt instanceof ASALTrigger) {
			return visit((ASALTrigger)evt);
		}
		
		if (evt instanceof ASALTimeout) {
			return visit((ASALTimeout)evt);
		}
		
		if (evt instanceof ASALCall) {
			return visit((ASALCall)evt);
		}
		
		throw new Error("Should not happen!");
	}
	
	public final T visitExpr(ASALExpr expr) throws ASALException {
		if (expr == null) {
			throw new Error("Should not happen!");
		}
		
		if (expr instanceof ASALBinaryExpr) {
			return visit((ASALBinaryExpr)expr);
		}
		
		if (expr instanceof ASALFunctionCall) {
			return visit((ASALFunctionCall)expr);
		}
		
		if (expr instanceof ASALLiteral) {
			return handle((ASALLiteral)expr);
		}
		
		if (expr instanceof ASALUnaryExpr) {
			return visit((ASALUnaryExpr)expr);
		}
		
		if (expr instanceof ASALVarRef) {
			return visit((ASALVarRef)expr);
		}
		
		throw new Error("Could not visit " + expr.getClass().getCanonicalName() + "!");
	}
	
	public final T visitStat(ASALStatement stat) throws ASALException {
		if (stat == null) {
			throw new Error("Should not happen!");
		}
		
		if (stat instanceof ASALAssignStatement) {
			return visit((ASALAssignStatement)stat);
		}
		
		if (stat instanceof ASALEmptyStatement) {
			return visit((ASALEmptyStatement)stat);
		}
		
		if (stat instanceof ASALFunctionCallStatement) {
			return visit((ASALFunctionCallStatement)stat);
		}
		
		if (stat instanceof ASALIfStatement) {
			return visit((ASALIfStatement)stat);
		}
		
		if (stat instanceof ASALWhileStatement) {
			return visit((ASALWhileStatement)stat);
		}
		
		if (stat instanceof ASALReturnStatement) {
			return visit((ASALReturnStatement)stat);
		}
		
		if (stat instanceof ASALSeqStatement) {
			return visit((ASALSeqStatement)stat);
		}
		
		throw new Error("Could not visit " + stat.getClass().getCanonicalName() + "!");
	}
	
	public final T visitFct(ASALFunction fct) throws ASALException {
		return handle(fct, visitStat(fct.getBody()));
	}
	
	public final List<T> visitExprs(List<? extends ASALExpr> exprs) throws ASALException {
		List<T> result = new ArrayList<T>();
		
		for (ASALExpr expr : exprs) {
			result.add(visitExpr(expr));
		}
		
		return result;
	}
}
