package lib.asal;

import java.util.*;

import lib.asal.parsing.ASALException;
import lib.asal.parsing.ASALParser;
import lib.asal.parsing.api.ASALAssignStatement;
import lib.asal.parsing.api.ASALStatement;
import lib.behave.Function;

public class ASALFunctionImpl implements ASALFunction {
	private Function declaration;
	private String name;
	private List<ASALVariable<?>> params;
	private ASALStatement body;
	private Set<ASALFunction> visitedFunctions;
	private Boolean isPure;
	
	public ASALFunctionImpl(String name, Function declaration) {
		this.name = name;
		this.declaration = declaration;
		
		params = declaration.createParams();
		visitedFunctions = null;
		isPure = null;
	}
	
	@Override
	public String getName() {
		return name;
	}
	
	@Override
	public List<ASALVariable<?>> getParams() {
		return params;
	}
	
	@Override
	public ASALDataType getReturnType() {
		return declaration.getReturnType();
	}
	
	@Override
	public Set<ASALFunction> getVisitedFunctions(ASALContext context) {
		if (visitedFunctions == null) {
			FunctionCallSearch search = new FunctionCallSearch(context);
			visitedFunctions = search.visitedFunctions;
		}
		
		return visitedFunctions;
	}
	
	@Override
	public boolean isPure(ASALContext context) {
		if (isPure == null) {
			FunctionCallSearch search = new FunctionCallSearch(context);
			visitedFunctions = search.visitedFunctions;
			isPure = search.isPure;
			
			if (isPure) {
				for (ASALFunction visitedFunction : visitedFunctions) {
					if (!visitedFunction.isPure(context)) {
						isPure = false;
						break;
					}
				}
			}
		}
		
		return isPure;
	}
	
	private static class FunctionCallSearch extends ASALVisitor<Object> {
		public final Set<ASALFunction> visitedFunctions;
		public boolean isPure;
		
		public FunctionCallSearch(ASALContext context) {
			super(context);
			
			visitedFunctions = new HashSet<ASALFunction>();
			isPure = true;
		}
		
		@Override
		public Object handle(ASALAssignStatement node, Object var, Object expr) {
			isPure = false;
			
			return super.handle(node, var, expr);
		}
		
		@Override
		public Object visitFctDef(ASALFunction fct) throws ASALException {
			visitedFunctions.add(fct);
			return null;
		}
	}
	
	@Override
	public void parse(ASALContextDecls context) throws ASALException {
		body = ASALParser.parseStatement(declaration.getCode());
		
		try {
			body.validate(declaration.createContext(context), false);
		} catch (ASALException e) {
			throw new ASALException("Could not validate function " + name + "():", declaration.getCode(), e);
		}
	}
	
	@Override
	public Function getDeclaration() {
		return declaration;
	}
	
	@Override
	public ASALStatement getBody() {
		return body;
	}
	
	public String getParamStr() {
		if (params.size() > 0) {
			String result = params.get(0) + ": " + params.get(0).getType().name;
			
			for (int index = 1; index < params.size(); index++) {
				result += ", " + params.get(index) + ": " + params.get(index).getType().name;
			}
			
			return result;
		}
		
		return "";
	}
	
	@Override
	public String toString() {
		return name + "(" + getParamStr() + "): " + getReturnType().name;
	}
	
	@Override
	public ASALContext createContext(ASALContext nestingContext) {
		ASALContext context = new ASALContext(nestingContext) {};
		
		for (ASALVariable<?> param : params) {
			context.fctParams.put(param.getName(), param);
		}
		
		return context;
	}
}


