%% Script that produces the table and figures for
% De Winter, J. C. F., & Dodou, D. (2021). Pitfalls of Statistical Methods in Traffic Psychology. Encyclopedia of Transportation.
%% Figure 1. Example (simulated) data for an experiment on automation-to-manual take-overs, with three time-budget conditions. A sample size of 20 per condition was assumed. The blue cross and the number next to each bar indicate the group mean.
clear all;close all;clc %#ok<CLALL>
opengl hardware
rng(1)
X(:,1)=1.4+1.4/3*randn(20,1); % Simulated data for time-budget = 3 s (20 participants)
X(:,2)=1.6+1.6/3*randn(20,1); % Simulated data for time-budget = 6 s (20 participants)
X(:,3)=2.0+2.0/3*randn(20,1); % Simulated data for time-budget = 9 s (20 participants)
notboxplot(X,2);grid on
set(gca,'TickDir','out','xticklabel',{'Time budget = 3 s' 'Time budget = 6 s' 'Time budget = 9 s'},'xtick',1:100,'xlim',[.6 3.4],'ylim',[0 3],'LooseInset',[0.01 0.01 0.01 0.01])
ylabel('Take-over time (s)')
h=findobj('FontName','Helvetica'); set(h,'FontSize',20,'Fontname','Arial')
disp(' ')
%% Figure 6. Example (simulated) data for an experiment on automation take-overs. The horizontal axis shows the take-over time (same data as the data in Figure 1 for the 3 s time budget condition), and the vertical axis shows the maximum absolute steering wheel angle.
figure
rng(6)
X=X(:,1); % take-over time
Y=X*3;Y=15+3*randn(size(Y)); % maximum absolute steering wheel angle
plot(X(:,1),Y(:,1),'ko','markersize',20,'linewidth',3);hold on
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
xlabel('Take-over time (s)')
ylabel('Maximum absolute steering wheel angle (deg)')
h=findobj('FontName','Helvetica'); set(h,'FontSize',20,'Fontname','Arial')
set(gcf,'pos',[1 41 1000 963])
[c,p]=corr(X,Y);
disp('Correlation and p-value for Figure 6')
disp([c p])
%% Figure 7. Example (simulated) data for an experiment on automation take-overs. The horizontal axis shows the take-over time, and the vertical axis shows the maximum absolute steering wheel angle. The data are the same as in Figure 6, except that each value is now repeated five times. A small variation in take-over time was added (within the range of -0.1 s to 0.1 s) to assure that the markers are not overlapping.
figure
X=[X(:,1);X(:,1)-0.1;X(:,1)-0.05;X(:,1)+0.05;X(:,1)+0.1]; % take-over time; same data as in Figure 6, except that each value is now repeated five times; a small variation in take-over time was added (within the range of -0.1 s to 0.1 s) to assure that the markers in the figure are not overlapping
Y=[Y(:,1);Y(:,1);Y(:,1);Y(:,1);Y(:,1)]; % maximum absolute steering wheel angle; same data as in Figure 6, except that each value is now repeated five times
plot(X(:,1),Y(:,1),'ko','markersize',20,'linewidth',3);hold on
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
xlabel('Take-over time (s)')
ylabel('Maximum absolute steering wheel angle (deg)')
h=findobj('FontName','Helvetica'); set(h,'FontSize',20,'Fontname','Arial')
set(gcf,'pos',[1 41 1000 963])
[c,p]=corr(X,Y);
disp('Correlation and p-value for Figure 7')
disp([c p])
%% Table 1. Results of a simulation study for detecting a significant difference between two groups in the presence of an outlier in one of the two groups.
rng(1)
reps=10^5;
[p,prank,p_outlier,prank_outlier,pnormal1,pnormal2,pnormal1_outlier,p_conditional,p_outlier_conditional]=deal(NaN(reps,1));
for i=1:reps
    A=randn(20,1); % 20 samples drawn from a population with a mean of 0 and a standard deviation of 1
    B=randn(20,1)+1; % 20 samples drawn from a population with a mean of 1 and a standard deviation of 1
    A(end)=4; % outlier in the first group (the value for the 20th participant in the first group equals 4)
    [~,p(i)]=ttest2(A,B); % compare the two samples with an independent-samples t-test
    prank(i)=ranksum(A,B); % compare the two samples with a Wilcoxon rank-sum test
    
    [~,pnormal1(i)]=swtest(A,0.05); % Shapiro-Wilk for the first group
    [~,pnormal2(i)]=swtest(B,0.05); % Shapiro-Wilk for the second group
    
    if pnormal1(i)>0.05 % compare the two samples with a conditional test
        p_conditional(i)=p(i); % when a Shapiro-Wilk test used on the first sample yielded p ≥ 0.05, an independent-samples t-test is conducted
    else 
        p_conditional(i)=prank(i); % perform Wilcoxon test when normality is violated
    end
    
end
disp('Table 1: Power t-test, Power rank sum test, Power conditional')
disp(round(100*[mean(p<0.05) mean(prank<0.05) mean(p_conditional<0.05)],1))
disp('Table 1: SW test')
disp(round(100*[mean(pnormal1<0.05) mean(pnormal2<0.05)],1))
%% Figure 3. Distribution of p-values (so-called p-curve) results when submitting a sample 1 (n = 20) and sample 2 (n = 20) to an independent-samples t-test, with Cohen’s d = 1. The vertical dashed lines are drawn at p = 0.04 and p = 0.05, respectively.
rng(1)
reps=10^5;
p=NaN(reps,1);
for i=1:reps
    x1=randn(20,1); % samples with n = 20 drawn from a population with a mean of 0 and a standard deviation of 1
    x2=randn(20,1)+0.8; % samples with n = 20 drawn from a population with a mean of 0.8 and a standard deviation of 1
    [~,p(i)]=ttest2(x1,x2); % compare the samples with an independent-samples t-test
end

V=0:0.001:1.00;
figure;hold on
D=histc(p,V);Dnorm=D;
Dnorm(end)=[];
V(end)=[];
plot(V+mean(diff(V))/2,Dnorm,'k-o','Linewidth',2);hold on
box on
xlabel('\itp\rm-value');ylabel('Number of \it{p}\rm-values in bin')
set(gca,'xlim',[0 1])
fig=gcf;set(findall(fig,'-property','FontSize'),'FontSize',20)
set(gca,'color','none','looseInset', [0.01 0.01 0.01 0.01])
plot([.05 .05],[0 18000], 'm:','Linewidth',2)
plot([.04 .04],[0 18000], 'm:','Linewidth',2)
disp('% of p-values between 0.04 and 0.05')
disp(100*mean(p>0.04 & p <0.05))
%% Figure 5. p-value distribution (p-curve) in case there is no effect in the population. If ‘trying out’ various statistical test and reporting the ‘best’ effect, the false positive rate may increase substantially (from 5.0% to 14.1% in this simulation).
rng('default')
reps=10^5;n=20;
[pp,p1,p2,p3,p4,p5,p6]=deal(NaN(reps,1));
for i=1:reps
    if rem(i/1000,1)==0;fprintf('Percentage completed = %5.3f',100*i/reps);fprintf('\n');end
    x=randn(n,1);y=randn(n,1); % two samples with equal means and standard deviations, and n = 20 per group
    [~,p1(i)]=ttest2(x,y); % independent-samples t-test
    [~,p2(i)]=ttest(x,y); % paired-samples t-test
    [~,p3(i)]=ttest2(rmoutliers(x),rmoutliers(y)); % independent-samples t-test after automated outlier removal
    [~,p4(i)]=ttest2(x(1:10),y(1:10)); % independent samples t-test for subgroups
    [~,p5(i)]=ttest2(x(11:20),y(11:20));
    pp(i)=min([p1(i) p2(i) p3(i) p4(i) p5(i)]); % choose the minimum p-value among all five options
end
%%
V=0:0.01:1.00;
figure;hold on
D1=histc(p1,V); %#ok<*HISTC>
D2=histc(pp,V);
D1(end)=[];
D2(end)=[];
V(end)=[];

plot(V+mean(diff(V))/2,D1,'k-o','Linewidth',2);hold on
plot(V+mean(diff(V))/2,D2,'r--o','Linewidth',2)
box on
legend('No effect - A single \it{t}\rm-test','No effect - Choosing the minimal \it{p}\rm-value from a variety of tests')
xlabel('\itp\rm-value');ylabel('Number of \it{p}\rm-values in bin')
set(gca,'xlim',[0 1])
fig=gcf;set(findall(fig,'-property','FontSize'),'FontSize',20)
set(gca,'color','none','looseInset', [0.01 0.01 0.01 0.01])
%% Figure 4. Distribution of Pearson product-moment correlation coefficients between two variables for a sample size of 300 when the correlation in the population is 0 (unbiased samples) and when the population correlation is 0 but response style contaminates the results (biased samples).
opengl hardware
clear all;clc %#ok<CLALL>
rng(1)
reps=10^5;
r0=NaN(reps,1);rc=r0;
for i=1:reps
    X=ceil(5*rand(300,2)); % measure two uncorrelated constructs using a five-point Likert scale and an unbiased sample of 300 respondents 
    temp=corrcoef(X); 
    r0(i)=temp(2,1); % correlations in the unbiased sample
    X(1:100,:)=X(1:100,:)+2; % one-third of the respondents is biased towards 'frequently'
    X(101:200,:)=X(101:200,:)-2; % one-third of the respondents is biased towards 'never'
    X(X<1)=1;
    X(X>5)=5;
    temp=corrcoef(X);
    rc(i)=temp(2,1); % correlations in the biased sample
end
V=-1:0.01:1;
D0=histc(r0,V);
Dc=histc(rc,V);
figure;hold on
plot(V+mean(diff(V)),D0,'k-o','Linewidth',3,'markersize',10)
plot(V+mean(diff(V)),Dc,'--s','color',[.5 .5 .5],'Linewidth',3,'markersize',10)
grid on
legend('Unbiased samples','Biased samples','location','northwest')
xlabel('Correlation coefficient');ylabel('Number of correlation coefficients in bin')
set(gca,'xlim',[-0.3 0.7])
fig=gcf;set(findall(fig,'-property','FontSize'),'FontSize',28)
set(gca,'looseInset', [0.01 0.01 0.01 0.01])
%% Figure 8. Example (simulated) data for vehicle speed and the driver’s vertical eye movements. These data were generated using a multisine approach and random number generator.
close all;clear all;clc %#ok<CLALL>
opengl software
rng(2); % rg should be 1, 2, 3, ... (this makes the random numbers repeatable)

Fs=50;              % sampling frequency (Hz)
L=1000;             % length of the measurement (s)
t=(0:L*Fs)*(1/Fs);  % time vector

cc=linspace(0.001,0.05,1000);
% Construct the signal
d = 999999999*rand(length(cc),1); % random phase shifts
yt = zeros(1,length(t));
for i = 1:length(cc) % loop for making the signal
    sinusoid=360*sin(2*pi*cc(i)*t+d(i)); % individual sinusoid with random phase shift
    yt=yt+sinusoid; % add to total signal
end
scaling_factor=5/(360*sqrt(length(cc))*sqrt(2)/2);
yt1=100+yt*scaling_factor; % normalize the total signal

cc2=linspace(0.001,0.2,1000); % frequency band
d2 = 999999999*rand(length(cc2),1); % random phase shifts
yt2 = zeros(1,length(t));
for i = 1:length(cc2) % loop for making the signal
    sinusoid=360*sin(2*pi*cc2(i)*t+d2(i)); % individual sinusoid with random phase shift
    yt2=yt2+sinusoid; % add to total signal
end
scaling_factor=10/(360*sqrt(length(cc2))*sqrt(2)/2);
yt2=yt2*scaling_factor; % normalize the total signal

corr(yt1(1:3000)',yt2(1:3000)')
figure
hold on
yyaxis left
plot(t,yt1,'Linewidth',5)
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'ylim',[85 110],'xlim',[0 60])
xlabel('Time (s)');ylabel('Vehicle speed (km/h)')
yyaxis right
plot(t,yt2,':','Linewidth',5)
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
ylabel('Vertical gaze angle (deg)')
%% Figure 2. Illustration of the effect of the pre-study probability that a research finding is true for statistics power (1−β) = 80%. For a significance level (α) of 5% and a pre-study probability of 5%, only 46% of the detected statistically significant effects are true positives.
RR=0:0.01:1;
power=0.8; 
alpha=0.05;
TP=power.*RR; % true positives
FP=alpha.*(1-RR); % false positives
PPV=TP./(TP+FP); % positive predictive value
figure;hold on;grid on
plot(100*RR,100*TP,'g','Linewidth',3)
plot(100*RR,100*FP,'r','Linewidth',3)
plot(100*RR,100*PPV,'k','Linewidth',3)
xlabel('Pre-study probability of finding being true  (%)' )
ylabel('Percentage of findings')
legend('True positives (TP)','False positives (FP)','Positive findings (\itp\rm < 0.05) that are true, TP/(TP+FP)')
h = findobj('FontName','Helvetica');
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
set(gca,'LooseInset',[0.01 0.01 0.01 0.01],'xtick',0:5:100)
