### Clear memory 
rm(list = ls()) 
### Load Apollo library 
library(apollo) 
### Initialise code 
apollo_initialise() 
### Set core controls 
apollo_control = list( modelName ="AV Time-use data", modelDescr ="MDCEV with outside good and constants only in utilities", indivID ="indivID" ) 

# ########## # #### LOAD DATA AND APPLY ANY TRANSFORMATIONS #### # ############ # 
setwd(dirname(rstudioapi::getActiveDocumentContext()$path)) #set working directory in RStudio
database = read.csv("Survey_data_MDCEV_stationary_model.csv",header=TRUE) 
database$outside = rowSums(database[,c("t_a05_s", "t_a06_s", "t_a08_s", "t_a09_s", "t_a10", "gaps_in_schedule","correction")])

############ # #### DEFINE MODEL PARAMETERS #### # ##############################
### # ### Vector of parameters, including any that are kept fixed in estimation 
apollo_beta = c(alpha_base = 0, 
                delta_outside = 0, 
                gamma_outside = 1,
                gamma_sleep = 1, 
                gamma_get_ready = 1, 
                gamma_work = 1, 
                gamma_meal = 1, 
                gamma_leisure = 1, 
                delta_sleep_work = 1, 
                delta_get_ready = 0, 
                delta_meal = 0, 
                delta_leisure = 0, 
                sigma = 1) 
### Vector with names (in quotes) of parameters to be kept fixed at their starting value in apollo_beta, use 
apollo_beta_fixed = c() ##if none 
apollo_fixed = c("delta_sleep_work","sigma","delta_outside","gamma_outside")


############ # #### GROUP AND VALIDATE INPUTS #### # ############################
apollo_inputs = apollo_validateInputs() 

############ # #### DEFINE MODEL AND LIKELIHOOD FUNCTION #### # #################
apollo_probabilities=function(apollo_beta, apollo_inputs, functionality="estimate")
  { ### Attach inputs and detach after function exit 
  apollo_attach(apollo_beta, apollo_inputs) 
  on.exit(apollo_detach(apollo_beta, apollo_inputs)) 
  ### Create list of probabilities P 
  P = list() 
  ### Define individual alternatives 
  alternatives = c("outside", "sleep", "get_ready", "work", "meal", "leisure") 
  ### Define availabilities 
  avail = list(outside = 1, sleep = 1, get_ready = 1, work = 1, meal = 1, leisure = 1) 
  ### Define continuous consumption for individual alternatives 
  continuousChoice = list(sleep = t_a01_s/60, get_ready = t_a02_s/60, work = t_a03_s/60, meal = t_a04_s/60, leisure = t_a07_s/60, outside = (t_a05_s + t_a06 + t_a08_s + t_a09_s + t_a10 + gaps_in_schedule + correction)/60) 
  ### Define utilities for individual alternatives 
  V = list() 
  V[["sleep" ]] = delta_sleep_work 
  V[["get_ready" ]] = delta_get_ready 
  V[["work" ]] = delta_sleep_work 
  V[["meal"]] = delta_meal
  V[["leisure" ]] = delta_leisure
  V[["outside"]] = delta_outside 
    ### Define alpha parameters 
  alpha = list(outside = alpha_base, sleep = alpha_base, get_ready = alpha_base, work = alpha_base, meal = alpha_base, leisure = alpha_base) 
  ### Define gamma parameters 
  gamma = list(sleep = gamma_sleep,
               get_ready = gamma_get_ready, 
               work = gamma_work,
               meal = gamma_meal,
               leisure = gamma_leisure,
               outside = gamma_outside) 
  ### Define costs for individual alternatives 
  cost = list(outside = 1, sleep = 1, get_ready = 1, work = 1, meal = 1, leisure = 1) 
  ### Define budget 
  budget = budget / 60
  ### Define settings for MDCEV model 
  mdcev_settings <- list(alternatives = alternatives, avail = avail, continuousChoice = continuousChoice, V = V, alpha = alpha, gamma = gamma, sigma = sigma, cost = cost, budget = budget) 
  ### Compute probabilities using MDCEV model 
  P[["model"]] = apollo_mdcev(mdcev_settings, functionality) 
  ### Take product across observation for same individual 
  P = apollo_panelProd(P, apollo_inputs, functionality) 
  ### Prepare and return outputs of function 
  P = apollo_prepareProb(P, apollo_inputs, functionality) 
  return(P) } 


################# # #### MODEL ESTIMATION #### # ########################## # 
model = apollo_estimate(apollo_beta, apollo_fixed, apollo_probabilities, apollo_inputs) 

################# # #### MODEL OUTPUTS #### # ############################# # 
#------ # #---- FORMATTED OUTPUT (TO SCREEN) ---- # ---------------------------- # 
apollo_modelOutput(model) 

#########---- FORMATTED OUTPUT (TO FILE, using model name) ---- ------------- # 
# apollo_saveOutput(model) 

# ################### # ##### POST-PROCESSING #### # ############ 
# ### Print outputs of additional diagnostics to new output file (remember to close file writing when complete) 
# sink(paste(model$apollo_control$modelName,"_additional_output.txt",sep=""),split=TRUE) 
# ------------------ # #---- MODEL PREDICTIONS ---- # -------------------- 
# ### Use the estimated model to make predictions 
# predictions_base = apollo_prediction(model, apollo_probabilities, apollo_inputs) 
# colMeans(predictions_base) # ----- # 
#---- switch off writing to file ---- ---------- # 
#if(sink.number()>0) 
#  sink()