### Clear memory 
rm(list = ls()) 
### Load Apollo library 
library(apollo) 
### Initialise code 
apollo_initialise() 
### Set core controls 
apollo_control = list( modelName ="AV Time-use data", modelDescr ="MDCEV without outside good and constants only in utilities", indivID ="indivID" ) 

# ########## # #### LOAD DATA AND APPLY ANY TRANSFORMATIONS #### # ############ # 
setwd(dirname(rstudioapi::getActiveDocumentContext()$path)) #set working directory in RStudio
database = read.csv("Survey_data_MDCEV_onboard_model.csv",header=TRUE) 
database$t_other = rowSums(database[,c("t_a01_o", "t_a05_o", "t_a06_o", "t_a08_o", "t_a09_o")])

############ # #### DEFINE MODEL PARAMETERS #### # ##############################
### # ### Vector of parameters, including any that are kept fixed in estimation 
apollo_beta = c(alpha_base = 0, 
                gamma_get_ready = 1, 
                gamma_work = 1, 
                gamma_meal = 1, 
                gamma_leisure = 1, 
                gamma_other = 1, 
                gamma_nothing = 1, 
                delta_get_ready = 0, 
                delta_work = 0, 
                delta_meal = 0, 
                delta_leisure = 0, 
                delta_other = 0, 
                delta_nothing = 0, 
                sigma = 1) 
### Vector with names (in quotes) of parameters to be kept fixed at their starting value in apollo_beta, use 
apollo_beta_fixed = c() ##if none 
apollo_fixed = c("sigma","delta_nothing")

############ # #### GROUP AND VALIDATE INPUTS #### # ############################
apollo_inputs = apollo_validateInputs() 

############ # #### DEFINE MODEL AND LIKELIHOOD FUNCTION #### # #################
apollo_probabilities=function(apollo_beta, apollo_inputs, functionality="estimate")
  { ### Attach inputs and detach after function exit 
  apollo_attach(apollo_beta, apollo_inputs) 
  on.exit(apollo_detach(apollo_beta, apollo_inputs)) 
  ### Create list of probabilities P 
  P = list() 
  ### Define individual alternatives 
  alternatives = c("get_ready", "meal", "work", "leisure", "other", "nothing") 
  ### Define availabilities 
  avail = list(get_ready = 1, meal = 1, work = 1, leisure = 1, other = 1, nothing = 1) 
  ### Define continuous consumption for individual alternatives 
  continuousChoice = list(get_ready = t_a02_o/60, work = t_a03_o/60, meal = t_a04_o/60, leisure = t_a07_o/60, other = (t_a01_o + t_a05_o + t_a06_o + t_a08_o + t_a09_o)/60, nothing = t_a11_o/60) 
  ### Define utilities for individual alternatives 
  V = list() 
  V[["get_ready" ]] = delta_get_ready 
  V[["work" ]] = delta_work 
  V[["meal" ]] = delta_meal 
  V[["leisure" ]] = delta_leisure 
  V[["other" ]] = delta_other
  V[["nothing" ]] = delta_nothing 
    ### Define alpha parameters 
  alpha = list(get_ready = 1 /(1 + exp(-alpha_base)), work = 1 /(1 + exp(-alpha_base)), meal = 1 /(1 + exp(-alpha_base)), leisure = 1 /(1 + exp(-alpha_base)), other = 1 /(1 + exp(-alpha_base)), nothing = 1 /(1 + exp(-alpha_base))) 
  ### Define gamma parameters 
  gamma = list(get_ready = gamma_get_ready,
               work = gamma_work, 
               meal = gamma_meal,
               leisure = gamma_leisure,
               other = gamma_other, 
               nothing = gamma_nothing) 
  ### Define costs for individual alternatives 
  cost = list(get_ready = 1, work = 1, meal = 1, leisure = 1, other = 1, nothing = 1) 
  ### Define budget 
  budget = t_a10/60
  ### Define settings for MDCEV model 
  mdcev_settings <- list(alternatives = alternatives, avail = avail, continuousChoice = continuousChoice, V = V, alpha = alpha, gamma = gamma, sigma = sigma, cost = cost, budget = budget) 
  ### Compute probabilities using MDCEV model 
  P[["model"]] = apollo_mdcev(mdcev_settings, functionality) 
  ### Take product across observation for same individual 
  P = apollo_panelProd(P, apollo_inputs, functionality) 
  ### Prepare and return outputs of function 
  P = apollo_prepareProb(P, apollo_inputs, functionality) 
  return(P) } 


################# # #### MODEL ESTIMATION #### # ########################## # 
model = apollo_estimate(apollo_beta, apollo_fixed, apollo_probabilities, apollo_inputs) 

################# # #### MODEL OUTPUTS #### # ############################# # 
#------ # #---- FORMATTED OUTPUT (TO SCREEN) ---- # ---------------------------- # 
apollo_modelOutput(model) 

#########---- FORMATTED OUTPUT (TO FILE, using model name) ---- ------------- # 
apollo_saveOutput(model) 

# ################### # ##### POST-PROCESSING #### # ############ 
# ### Print outputs of additional diagnostics to new output file (remember to close file writing when complete) 
# sink(paste(model$apollo_control$modelName,"_additional_output.txt",sep=""),split=TRUE) 
# ------------------ # #---- MODEL PREDICTIONS ---- # -------------------- 
# ### Use the estimated model to make predictions 
# predictions_base = apollo_prediction(model, apollo_probabilities, apollo_inputs) 
# colMeans(predictions_base) # ----- # 
#---- switch off writing to file ---- ---------- # 
#if(sink.number()>0) 
#  sink()