# -*- coding: utf-8 -*-

"""
Module with class TicToc to replicate the functionality of MATLAB's tic and toc.

Documentation: https://pypi.python.org/pypi/pytictoc
"""

__author__       = 'Eric Fields'
__version__      = '1.5.2'
__version_date__ = '23 April 2021'


from timeit import default_timer
from datetime import timedelta

class TicToc(object):
    
    """
    Replicate the functionality of MATLAB's tic and toc.
    
    #Methods
    TicToc.tic()       #start or re-start the timer
    TicToc.toc()       #print elapsed time since timer start
    TicToc.tocvalue()  #return floating point value of elapsed time since timer start
    
    #Attributes
    TicToc.start     #Time from timeit.default_timer() when t.tic() was last called
    TicToc.end       #Time from timeit.default_timer() when t.toc() or t.tocvalue() was last called
    TicToc.elapsed   #t.end - t.start; i.e., time elapsed from t.start when t.toc() or t.tocvalue() was last called
    """
    
    def __init__(self):
        """Create instance of TicToc class."""
        self.start   = float('nan')
        self.end     = float('nan')
        self.elapsed = float('nan')
        self.last = float('nan')
        self.count = 0
        self.average = float('nan')
        
    def tic(self):
        """Start the timer."""
        self.start = default_timer()
        self.last = self.start
        self.count = 0
        
    def toc(self, msg='Elapsed time is', restart=False):
        """
        Report time elapsed since last call to tic().
        
        Optional arguments:
            msg     - String to replace default message of 'Elapsed time is'
            restart - Boolean specifying whether to restart the timer
        """
        self.end = default_timer()
        self.elapsed = self.end - self.start
        t_elaspd = timedelta(seconds=self.elapsed)
        v_delta = self.end - self.last
        self.last = self.end
        t_delta = timedelta(seconds=v_delta)
#        print('%s %f seconds.' % (msg, self.elapsed))
        print('{}: {}\t{}'.format(msg, str(t_elaspd)[:-3], str(t_delta)[:-3]))
#        print('{}: {}'.format(msg, str(t_elaspd)[:-3] ))
        if restart:
#            self.start = default_timer()
            self.tic()
        
    def tocvalue(self, restart=False):
        """
        Return time elapsed since last call to tic().
        
        Optional argument:
            restart - Boolean specifying whether to restart the timer
        """
        self.end     = default_timer()
        self.elapsed = self.end - self.start
        if restart:
#            self.start = default_timer()
            self.tic()
        return self.elapsed

    def tocstate(self):
        self.end     = default_timer()
        self.elapsed = self.end - self.start
        t_last = self.end - self.last
        self.last = self.end
        self.count += 1
        self.average = self.elapsed / self.count
        return self.elapsed, t_last, self.average
    
    def __enter__(self):
        """Start the timer when using TicToc in a context manager."""
        self.start = default_timer()
        self.tic()
    
    def __exit__(self, *args):
        """On exit, pring time elapsed since entering context manager."""
        self.end = default_timer()
        self.elapsed = self.end - self.start
        print('Elapsed time is %f seconds.' % self.elapsed)
